package to.then.kie;

import com.amazonaws.services.s3.AmazonS3;
import java.io.File;
import java.net.URISyntaxException;
import java.util.Collection;
import org.apache.log4j.Logger;
import org.apache.maven.repository.internal.MavenRepositorySystemUtils;
import org.apache.maven.wagon.Wagon;
import org.drools.compiler.kie.builder.impl.AbstractKieModule;
import org.drools.compiler.kproject.ReleaseIdImpl;
import org.eclipse.aether.artifact.Artifact;
import org.eclipse.aether.artifact.DefaultArtifact;
import org.eclipse.aether.collection.CollectRequest;
import org.eclipse.aether.connector.basic.BasicRepositoryConnectorFactory;
import org.eclipse.aether.deployment.DeployRequest;
import org.eclipse.aether.deployment.DeploymentException;
import org.eclipse.aether.graph.Dependency;
import org.eclipse.aether.impl.DefaultServiceLocator;
import org.eclipse.aether.internal.impl.DefaultRepositorySystem;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.resolution.ArtifactRequest;
import org.eclipse.aether.resolution.ArtifactResolutionException;
import org.eclipse.aether.resolution.DependencyRequest;
import org.eclipse.aether.resolution.DependencyResolutionException;
import org.eclipse.aether.spi.connector.RepositoryConnectorFactory;
import org.eclipse.aether.spi.connector.transport.TransporterFactory;
import org.eclipse.aether.transport.file.FileTransporterFactory;
import org.eclipse.aether.transport.http.HttpTransporterFactory;
import org.eclipse.aether.transport.wagon.WagonProvider;
import org.eclipse.aether.transport.wagon.WagonTransporterFactory;
import org.eclipse.aether.util.artifact.SubArtifact;
import org.kie.api.KieServices;
import org.kie.api.builder.ReleaseId;
import org.kie.scanner.Aether;
import org.kie.scanner.MavenRepository;
import static org.kie.scanner.MavenRepository.toFileName;
import to.then.maven.wagon.s3.AmazonS3Wagon;

public class KieRepository {

    private final static Logger log = Logger.getLogger(KieRepository.class);
    private final KieServices kie;
    private final Aether aether;

    public KieRepository(AmazonS3 s3) {
        aether = initAether(s3);
        kie = KieServices.Factory.get();
    }

    public AbstractKieModule getKieModule(ReleaseId releaseId, Collection<RemoteRepository> repositories) {
        resolveDependencies(releaseId.toExternalForm(), repositories);
        return (AbstractKieModule) kie.getRepository().getKieModule(releaseId);
    }

    public void putKieModule(AbstractKieModule kmodule, RemoteRepository repository) {

        ReleaseId releaseId = kmodule.getReleaseId();
        byte[] jarBytes = kmodule.getBytes();
        byte[] pomBytes = kmodule.getBytes(((ReleaseIdImpl) releaseId).getPomXmlPath());

        MavenRepository.getMavenRepository().deployArtifact(releaseId, jarBytes, pomBytes);

        File jar = new File(System.getProperty("java.io.tmpdir"), toFileName(releaseId, null) + ".jar");
        File pom = new File(System.getProperty("java.io.tmpdir"), toFileName(releaseId, null) + ".pom");

        Artifact jarArtifact = new DefaultArtifact(releaseId.getGroupId(), releaseId.getArtifactId(), "jar", releaseId.getVersion());
        jarArtifact = jarArtifact.setFile(jar);

        Artifact pomArtifact = new SubArtifact(jarArtifact, "", "pom");
        pomArtifact = pomArtifact.setFile(pom);

        DeployRequest deployRequest = new DeployRequest();
        deployRequest
                .addArtifact(jarArtifact)
                .addArtifact(pomArtifact)
                .setRepository(repository);

        try {
            Aether.getAether().getSystem().deploy(Aether.getAether().getSession(), deployRequest);
        } catch (DeploymentException e) {
            throw new RuntimeException(e);
        }
    }

    private Aether initAether(AmazonS3 s3) {
        try {
            System.setProperty("kie.maven.settings.custom", KieRepository.class.getResource("/settings.xml").toURI().getPath());
        } catch (URISyntaxException ex) {
            throw new RuntimeException(ex);
        }
        Aether aether = Aether.getAether();
        DefaultRepositorySystem system = (DefaultRepositorySystem) aether.getSystem();
        DefaultServiceLocator locator = MavenRepositorySystemUtils.newServiceLocator();
        locator.addService(RepositoryConnectorFactory.class, BasicRepositoryConnectorFactory.class);
        locator.addService(TransporterFactory.class, FileTransporterFactory.class);
        locator.addService(TransporterFactory.class, HttpTransporterFactory.class);
        locator.addService(TransporterFactory.class, WagonTransporterFactory.class);
        locator.setServices(WagonProvider.class, new WagonProvider() {
            @Override
            public Wagon lookup(String protocol) throws Exception {
                if ("s3".equals(protocol)) {
                    return new AmazonS3Wagon(s3);
                }
                return null;
            }

            @Override
            public void release(Wagon wagon) {
            }
        });
        system.initService(locator);
        return aether;
    }

    public void resolveDependencies(String dependency, Collection<RemoteRepository> repositories) {
        Artifact artifact = new DefaultArtifact( dependency );
        CollectRequest collectRequest = new CollectRequest();
        Dependency root = new Dependency( artifact, "" );
        collectRequest.setRoot( root );
        DependencyRequest dependencyRequest = new DependencyRequest();
        dependencyRequest.setCollectRequest(collectRequest);
        repositories.stream().forEach((repo) -> {
            collectRequest.addRepository(repo);
        });
        try {
            aether.getSystem().resolveDependencies(aether.getSession(), dependencyRequest);
        } catch (DependencyResolutionException e) {
            log.warn("Unable to resolve dependency: " + dependency);
        }
    }
    
    public void resolveArtifact(String dependency, Collection<RemoteRepository> repositories) {
        Artifact artifact = new DefaultArtifact(dependency);
        ArtifactRequest artifactRequest = new ArtifactRequest();
        artifactRequest.setArtifact(artifact);
        repositories.stream().forEach((repo) -> {
            artifactRequest.addRepository(repo);
        });
        try {
            aether.getSystem().resolveArtifact(aether.getSession(), artifactRequest);
        } catch (ArtifactResolutionException e) {
            log.warn("Unable to resolve artifact: " + dependency);
        }
    }
}
