package top.doudou.base.util.file;

import lombok.extern.slf4j.Slf4j;
import top.doudou.base.exception.CustomException;
import top.doudou.base.stream.StreamCloseUtils;

import java.io.*;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * @Description 文件工具类
 * @Author 傻男人 <244191347@qq.com>
 * @Date 2020-11-16 14:34
 * @Version V1.0
 */
@Slf4j
public class FileUtil {

    /**
     * 文件是否存在
     * @param file
     * @return
     */
    public static boolean exist(File file){
        return file.exists();
    }

    public static boolean exist(String filePath){
        return new File(filePath).exists();
    }

    /**
     * 文件是否是文件夹
     * @param file
     * @return
     */
    public static boolean isDirectory(File file){
        return file.isDirectory();
    }

    public static boolean isDirectory(String filePath){
        return new File(filePath).isDirectory();
    }

    public static File getParentFile(File file){
        return file.getParentFile();
    }

    /**
     * 创建一个新的文件，假如文件夹不存在，一并创建文件夹
     * @param file
     * @return
     * @throws IOException
     */
    public static boolean createNewFile(File file) throws IOException {
        if(exist(file)){
            return file.createNewFile();
        }
        if(isDirectory(file)){
            return file.mkdirs();
        }
        File parentFile = file.getParentFile();
        parentFile.mkdirs();
        return file.createNewFile();
    }

    /**
     * 创建文件或者文件夹，假如存在则不创建新的文件
     * @param file
     * @return
     * @throws IOException
     */
    public static boolean touch(File file) throws IOException {
        if(exist(file)){
            return true;
        }
        if(isDirectory(file)){
            return file.mkdirs();
        }
        File parentFile = file.getParentFile();
        if(!exist(file)){
            boolean mkdirs = parentFile.mkdirs();
            if(!mkdirs){
                throw new CustomException("创建父目录失败，目录为：{}",parentFile.getPath());
            }
        }
        return file.createNewFile();
    }

    /**
     * 获取文件的后缀,扩展名不带“.”
     * @param file
     * @return
     */
    public static String getSuffix(File file){
        if(isDirectory(file)){
            throw new CustomException("文件为文件夹，不能获取文件的后缀");
        }
        String fileName = file.getName();
        if (fileName == null) {
            return null;
        }
        if(fileName.length() == 0){
            return fileName;
        }
        int index = fileName.lastIndexOf(".");
        if (index == -1) {
            return "";
        } else {
            return fileName.substring(index + 1);
        }
    }

    public static String getSuffix(String filePath){
        return getSuffix(new File(filePath));
    }

    /**
     * 获取文件的前缀
     * @param file
     * @return
     */
    public static String getPrefix(File file){
        String fileName = file.getName();
        if (file.isDirectory()) {
            return fileName;
        }
        if (null == fileName) {
            return null;
        }
        int len = fileName.length();
        if (0 == len) {
            return fileName;
        }
        int index = fileName.lastIndexOf(".");
        if (index == -1) {
            return fileName;
        } else {
            return fileName.substring(0,index);
        }
    }

    public static String getPrefix(String filePath){
        return getPrefix(new File(filePath));
    }

    /**
     * 文件复制
     * @param sourceFile 源文件
     * @param targetFile 目标文件
     * @return
     */
    public static boolean copy(File sourceFile, File targetFile) {
        boolean success = true;
        FileInputStream in = null;
        FileOutputStream out = null;
        try {
            in = new FileInputStream(sourceFile);
            out = new FileOutputStream(targetFile);
            byte[] buffer = new byte[1024];
            int len = 0;
            while ((len = in.read(buffer)) > 0) {
                out.write(buffer);
            }
        } catch (FileNotFoundException e) {
            success = false;
        } catch (IOException e) {
            success = false;
        }finally {
            StreamCloseUtils.close(in);
            StreamCloseUtils.close(out);
        }
        return success;
    }

    /**
     * 压缩文件
     * @param srcFiles  需要压缩的文件组
     * @param zipFile  压缩后的文件
     */
    public static void zipFiles(List<File> srcFiles, File zipFile) {
        // 判断压缩后的文件存在不，不存在则创建
        if (!zipFile.exists()) {
            try {
                zipFile.createNewFile();
            } catch (IOException e) {
                throw new CustomException("压缩后的文件不存在，创建压缩后的文件失败");
            }
        }
        // 创建 FileOutputStream 对象
        FileOutputStream fileOutputStream = null;
        // 创建 ZipOutputStream
        ZipOutputStream zipOutputStream = null;
        // 创建 FileInputStream 对象
        FileInputStream fileInputStream = null;
        try {
            // 实例化 FileOutputStream 对象
            fileOutputStream = new FileOutputStream(zipFile);
            // 实例化 ZipOutputStream 对象
            zipOutputStream = new ZipOutputStream(fileOutputStream);
            // 创建 ZipEntry 对象
            ZipEntry zipEntry = null;
            // 遍历源文件数组
            for (int i = 0; i < srcFiles.size(); i++) {
                // 将源文件数组中的当前文件读入 FileInputStream 流中
                File item = srcFiles.get(i);
                fileInputStream = new FileInputStream(item);
                // 实例化 ZipEntry 对象，源文件数组中的当前文件
                zipEntry = new ZipEntry(item.getName());
                zipOutputStream.putNextEntry(zipEntry);
                // 该变量记录每次真正读的字节个数
                int len;
                // 定义每次读取的字节数组
                byte[] buffer = new byte[1024];
                while ((len = fileInputStream.read(buffer)) > 0) {
                    zipOutputStream.write(buffer, 0, len);
                }
            }

        } catch (IOException e) {
            e.printStackTrace();
        }finally {
            try {
                zipOutputStream.closeEntry();
            } catch (IOException ioException) {
                ioException.printStackTrace();
            }
            StreamCloseUtils.close(zipOutputStream);
            StreamCloseUtils.close(fileInputStream);
            StreamCloseUtils.close(fileOutputStream);
        }

    }
}
