package top.doudou.base.util;

import org.springframework.lang.Nullable;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import top.doudou.base.exception.ValidateException;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

/**
 * @author  傻男人<244191347@qq.com>
 * @Date: 2020/5/15 9:36
 * @Version: 1.0
 * @Description: 断言
 */
public class FastAssert {

    public FastAssert() {
    }

    /**
     * 断言 为true
     * @param expression
     * @param message
     */
    public static void state(boolean expression, String message) {
        if (!expression) {
            throw new ValidateException(message);
        }
    }

    /**
     * 断言 为true
     * @param expression
     * @param message
     */
    public static void isTrue(boolean expression, String message) {
        state(expression,message);
    }

    /**
     * 对象为空
     * @param object
     * @param message
     */
    public static void isNull(@Nullable Object object, String message) {
        if (object != null) {
            throw new ValidateException(message);
        }
    }

    /**
     *  对象不为空
     * @param obj
     * @param message
     */
    public static void notNull(@Nullable Object obj, String message) {
        if (null == obj) {
            throw new ValidateException(message);
        }
    }

    /**
     *  对象不为空
     * @param obj
     * @param message
     */
    public static void notEmpty(@Nullable Object obj, String message) {
        notNull(obj,message);
        if(obj instanceof String && StringUtils.isEmpty(obj)){
            throw new ValidateException(message);
        }
        if(obj instanceof Collection && CollectionUtils.isEmpty((Collection)obj)){
            throw new ValidateException(message);
        }
        if(obj instanceof Map && CollectionUtils.isEmpty((Map)obj)){
            throw new ValidateException(message);
        }
    }

    /**
     * 对象有长度
     * @param text
     * @param message
     */
    public static void hasLength(@Nullable String text, String message) {
        if (!StringUtils.hasLength(text)) {
            throw new ValidateException(message);
        }
    }


    public static void hasText(@Nullable String text, String message) {
        if (!StringUtils.hasText(text)) {
            throw new ValidateException(message);
        }
    }

    /**
     * 不包含后面内容
     * @param textToSearch
     * @param substring
     * @param message
     */
    public static void doesNotContain(@Nullable String textToSearch, String substring, String message) {
        if (StringUtils.hasLength(textToSearch) && StringUtils.hasLength(substring) && textToSearch.contains(substring)) {
            throw new ValidateException(message);
        }
    }

    /**
     * 没有空元素
     * @param array
     * @param message
     */
    public static void noNullElements(@Nullable Object[] array, String message) {
        if (array != null) {
            Object[] var2 = array;
            int var3 = array.length;

            for(int var4 = 0; var4 < var3; ++var4) {
                Object element = var2[var4];
                if (element == null) {
                    throw new ValidateException(message);
                }
            }
        }

    }


    /**
     * 不包含空元素
     * @param collection
     * @param message
     */
    public static void noNullElements(@Nullable Collection<?> collection, String message) {
        if (collection != null) {
            Iterator var2 = collection.iterator();

            while(var2.hasNext()) {
                Object element = var2.next();
                if (element == null) {
                    throw new ValidateException(message);
                }
            }
        }

    }


    public static void isInstanceOf(Class<?> type, @Nullable Object obj, String message) {
        notNull(type, (String)"Type to check against must not be null");
        if (!type.isInstance(obj)) {
            instanceCheckFailed(type, obj, message);
        }

    }

    public static void isInstanceOf(Class<?> type, @Nullable Object obj) {
        isInstanceOf(type, obj, "");
    }

    public static void isAssignable(Class<?> superType, @Nullable Class<?> subType, String message) {
        notNull(superType, (String)"Super type to check against must not be null");
        if (subType == null || !superType.isAssignableFrom(subType)) {
            assignableCheckFailed(superType, subType, message);
        }

    }

    public static void isAssignable(Class<?> superType, Class<?> subType) {
        isAssignable(superType, subType, "");
    }

    private static void instanceCheckFailed(Class<?> type, @Nullable Object obj, @Nullable String msg) {
        String className = obj != null ? obj.getClass().getName() : "null";
        String result = "";
        boolean defaultMessage = true;
        if (StringUtils.hasLength(msg)) {
            if (endsWithSeparator(msg)) {
                result = msg + " ";
            } else {
                result = messageWithTypeName(msg, className);
                defaultMessage = false;
            }
        }

        if (defaultMessage) {
            result = result + "Object of class [" + className + "] must be an instance of " + type;
        }

        throw new IllegalArgumentException(result);
    }

    private static void assignableCheckFailed(Class<?> superType, @Nullable Class<?> subType, @Nullable String msg) {
        String result = "";
        boolean defaultMessage = true;
        if (StringUtils.hasLength(msg)) {
            if (endsWithSeparator(msg)) {
                result = msg + " ";
            } else {
                result = messageWithTypeName(msg, subType);
                defaultMessage = false;
            }
        }

        if (defaultMessage) {
            result = result + subType + " is not assignable to " + superType;
        }

        throw new IllegalArgumentException(result);
    }

    private static boolean endsWithSeparator(String msg) {
        return msg.endsWith(":") || msg.endsWith(";") || msg.endsWith(",") || msg.endsWith(".");
    }

    private static String messageWithTypeName(String msg, @Nullable Object typeName) {
        return msg + (msg.endsWith(" ") ? "" : ": ") + typeName;
    }

}