package top.doudou.base.api.version;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.servlet.mvc.condition.RequestCondition;

import javax.servlet.http.HttpServletRequest;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @Description 接口多版本处理
 * @version: 1.0
 * @Created 傻男人 <244191347@qq.com>
 * @Date 2021-10-22 17:17
 */
@Slf4j
public class ApiVersionRequestCondition implements RequestCondition<ApiVersionRequestCondition> {

    /**
     * Api 版本参数名称
     */
    public final static String VERSION_PARAM_NAME = "api-version";

    /**
     * 路径中版本的前缀， 这里用 /v[1-9]/的形式
     */
    private final static Pattern VERSION_PREFIX_PATTERN = Pattern.compile("v(\\d+)\\.(\\d+)");

    /**
     * 默认版本号（未添加注解的版本号）
     */
    public static final ApiVersionRequestCondition DEFAULT_API_VERSION = new ApiVersionRequestCondition(1.0);

    public static final double DEFAULT_VERSION = 1.0;

    private final double version;

    public ApiVersionRequestCondition(double version){
        this.version = version;
    }

    public double getVersion() {
        return version;
    }

    @Override
    public ApiVersionRequestCondition combine(ApiVersionRequestCondition other) {
        // 采用最后定义优先原则，则方法上的定义覆盖类上面的定义
        return new ApiVersionRequestCondition(other.getVersion());
    }

    /**
     * 版本匹配
     * @param request
     * @return
     */
    @Override
    public ApiVersionRequestCondition getMatchingCondition(HttpServletRequest request) {
        String v = getRequestVersion(request);
        Matcher m = VERSION_PREFIX_PATTERN.matcher(v + ".999");
        if(m.find()){
            v = m.group(1) + "." + m.group(2);
            log.debug("---->  [ApiVersionRequestCondition-getMatchingCondition]  v:{}   this:{}",v,this.version);
            // 如果请求的版本号大于配置版本号， 则满足
            if(Double.parseDouble(v) >= this.version) {
                return this;
            }
        }
        return null;
    }

    @Override
    public int compareTo(ApiVersionRequestCondition other, HttpServletRequest httpServletRequest) {
        // 优先匹配最新的版本号
        return Double.compare(other.getVersion(), this.version);
    }

    /**
     * 获取请求的版本号
     * @param request
     * @return
     */
    public String getRequestVersion(HttpServletRequest request){
        String paramVersion = request.getHeader(VERSION_PARAM_NAME);
        if(paramVersion==null){
            paramVersion = request.getParameter(VERSION_PARAM_NAME);
        }
        //未找到版本号，则默认最低版本1.0
        if(StringUtils.isNotEmpty(paramVersion)){
            if(Double.parseDouble(paramVersion) <= DEFAULT_VERSION){
                paramVersion = String.valueOf(DEFAULT_VERSION);
            }
        }else {
            paramVersion = "v1.0";
        }
        if(!paramVersion.startsWith("v")){
            paramVersion = "v" + paramVersion;
        }
        return paramVersion;
    }
}
