package top.doudou.base.context;

import org.springframework.beans.BeansException;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.stereotype.Component;

import java.lang.annotation.Annotation;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @Description 上下文工具类(需要先设置ApplicationContext)
 * @Author 傻男人 <244191347@qq.com>
 * @Date 2020-10-15 10:10
 * @Version V1.0
 */
@Component
public class ApplicationContextUtils implements ApplicationContextAware {

    /** Spring应用上下文环境*/
    public static ApplicationContext context;

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        ApplicationContextUtils.context = applicationContext;
    }

    /**
     * 获取上下文
     * @return
     */
    public static synchronized ApplicationContext getContext() {
        return context;
    }

    /**
     * 实现ApplicationContextAware接口的回调方法，设置上下文环境
     * @param context
     */
    public synchronized static void setContext(ApplicationContext context) {
        ApplicationContextUtils.context = context;
    }

    /**
     * 通过名字获取上下文中的bean
     * @param name
     * @return Object 一个以所给名字注册的bean的实例
     * @return
     */
    public static Object getBean(String name){
        try{
            return context.getBean(name);
        }catch (NoSuchBeanDefinitionException e){
            return null;
        }
    }

    /**
     *获取类型为requiredType的对象 如果bean不能被类型转换，相应的异常将会被抛出（BeanNotOfRequiredTypeException）
     * @param name       bean注册名
     * @param requiredType 返回对象类型
     * @return
     */
    public static <T>T getBean(String name,Class<T> requiredType){
        try{
            return (T) context.getBean(name);
        }catch (Exception e){
            return null;
        }
    }

    /**
     * 通过类型获取上下文中的bean
     * @param requiredType
     * @return
     */
    public static <T>T getBean(Class<T> requiredType){
        return context.getBean(requiredType);
    }

    /**
     * 如果BeanFactory包含一个与所给名称匹配的bean定义，则返回true
     * @param name
     * @return boolean
     */
    public static boolean containsBean(String name) {
        return context.containsBean(name);
    }

    /**
     * 判断以给定名字注册的bean定义是一个singleton还是一个prototype。
     * 如果与给定名字相应的bean定义没有被找到，将会抛出一个异常（NoSuchBeanDefinitionException）
     * @param name
     * @return boolean
     * @throws NoSuchBeanDefinitionException
     */
    public static boolean isSingleton(String name) throws NoSuchBeanDefinitionException {
        return context.isSingleton(name);
    }

    /**
     * @param name
     * @return Class 注册对象的类型
     * @throws NoSuchBeanDefinitionException
     */
    public static Class getType(String name)
            throws NoSuchBeanDefinitionException {
        return context.getType(name);
    }

    /**
     * 如果给定的bean名字在bean定义中有别名，则返回这些别名
     * @param name
     * @return
     * @throws NoSuchBeanDefinitionException
     */
    public static String[] getAliases(String name)
            throws NoSuchBeanDefinitionException {
        return context.getAliases(name);
    }

    /**
     * 获取当前环境
     * @return
     */
    public static String[] getActiveProfile() {
        return context.getEnvironment().getActiveProfiles();
    }

    /**
     * 获取指定配置的属性
     * @param key
     * @return
     */
    public static String getProperty(String key){
        return context.getEnvironment().getProperty(key);
    }
    /**
     * 获取指定配置的属性
     * @param key
     * @return
     */
    public static <T>T getProperty(String key,Class<T> target){
        return context.getEnvironment().getProperty(key,target);
    }


    public static <T> T registerBean(String beanName,Class<T> requiredType){
        T t = ApplicationContextUtils.getBean(beanName,requiredType);
        if(t == null){
            //将applicationContext转换为ConfigurableApplicationContext
            ConfigurableApplicationContext configurableApplicationContext = (ConfigurableApplicationContext) ApplicationContextUtils.getContext();

            // 获取bean工厂并转换为DefaultListableBeanFactory
            DefaultListableBeanFactory defaultListableBeanFactory = (DefaultListableBeanFactory) configurableApplicationContext.getBeanFactory();

            // 通过BeanDefinitionBuilder创建bean定义
            BeanDefinitionBuilder beanDefinitionBuilder = BeanDefinitionBuilder.genericBeanDefinition(requiredType);

            // 注册bean
            defaultListableBeanFactory.registerBeanDefinition(beanName, beanDefinitionBuilder.getRawBeanDefinition());

            t = ApplicationContextUtils.getBean(beanName,requiredType);
        }
        return t;
    }

    public static List<Object> getBeansWithAnnotation(Class<? extends Annotation> annotationType) {
        Map<String, Object> beansWithAnnotation = context.getBeansWithAnnotation(annotationType);
        return beansWithAnnotation.entrySet().stream().map(entry->entry.getValue()).collect(Collectors.toList());
    }
}
