package top.doudou.base.context;

import lombok.extern.slf4j.Slf4j;
import org.slf4j.MDC;
import org.springframework.boot.context.event.ApplicationEnvironmentPreparedEvent;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.event.GenericApplicationListener;
import org.springframework.core.Ordered;
import org.springframework.core.ResolvableType;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.MutablePropertySources;
import org.springframework.core.env.PropertySource;
import org.springframework.util.StringUtils;

import java.io.File;

/**
 * @Description 自定义的一个SpringBoot的事件监听对象
 * 启动事件的顺序
 * ApplicationStartingEvent-这个事件在 Spring Boot 应用运行开始时，且进行任何处理之前发送（除了监听器和初始化器注册之外）
 * ApplicationEnvironmentPreparedEvent-这个事件在当已知要在上下文中使用 Spring 环境（Environment）时，在 Spring 上下文（context）创建之前发送
 * ApplicationContextInitializedEvent-这个事件在当 Spring 应用上下文（ApplicationContext）准备好了，并且应用初始化器（ApplicationContextInitializers）已经被调用，在 bean 的定义（bean definitions）被加载之前发送
 * WebServerInitializedEvent-这个 Web 服务器初始化事件在 WebServer 启动之后发送，对应的还有 ServletWebServerInitializedEvent（Servlet Web 服务器初始化事件）、ReactiveWebServerInitializedEvent（响应式 Web 服务器初始化事件）。
 * ContextRefreshedEvent-这个上下文刷新事件是在 Spring 应用上下文（ApplicationContext）刷新之后发送
 * ApplicationPreparedEvent-这个事件是在 Spring 上下文（context）刷新之前，且在 bean 的定义（bean definitions）被加载之后发送
 * ApplicationStartedEvent-这个事件是在 Spring 上下文（context）刷新之后，且在 application/ command-line runners 被调用之前发送
 * AvailabilityChangeEvent-这个事件紧随上个事件之后发送，状态：ReadinessState.CORRECT，表示应用已处于活动状态
 * ApplicationReadyEvent-这个事件在任何 application/ command-line runners 调用之后发送
 * AvailabilityChangeEvent-这个事件紧随上个事件之后发送，状态：ReadinessState.ACCEPTING_TRAFFIC，表示应用可以开始准备接收请求了
 * ApplicationFailedEvent-这个事件在应用启动异常时进行发送
 * @Author 傻男人 <244191347@qq.com>
 * @Date 2020-10-16 11:10
 * @Version V1.0
 */
@Slf4j
public class ApplicationStartEventListener implements GenericApplicationListener {

    public static final int DEFAULT_ORDER = Ordered.HIGHEST_PRECEDENCE + 10;

    private static Class<?>[] EVENT_TYPES = {
            ApplicationEnvironmentPreparedEvent.class};

    private static final String APPLICATION_NAME = "spring.application.name";

    private static final String BASE_PATH = "custom.log.base-path";

    private static final String LOG_NAME = "custom.log.log-name";

    @Override
    public void onApplicationEvent(ApplicationEvent event) {
        ConfigurableEnvironment envi = ((ApplicationEnvironmentPreparedEvent) event).getEnvironment();
        MutablePropertySources mps = envi.getPropertySources();

        //从多个配置源对象中，获得系统属性配置对象
        PropertySource<?> ps = mps.get("configurationProperties");

        if (ps != null) {
            String applicationName = (String) ps.getProperty(APPLICATION_NAME);
            String basePath = (String) ps.getProperty(BASE_PATH);
            String logName = (String) ps.getProperty(LOG_NAME);
            log.debug("日志文件存储的位置："+basePath+"  日志文件的名字："+logName);
            if(StringUtils.isEmpty(basePath)){
                basePath = System.getProperty("user.home");
            }
            if(StringUtils.isEmpty(logName)){
                logName = "output.log";
            }
            MDC.put("logName", logName);
            MDC.put("logPath", basePath+ File.separator+applicationName);
        }
    }


    @Override
    public int getOrder() {
        return DEFAULT_ORDER;
    }

    @Override
    public boolean supportsEventType(ResolvableType resolvableType) {
        boolean assignableFrom = isAssignableFrom(resolvableType, EVENT_TYPES);
        if(assignableFrom){
            log.debug("匹配成功的事件的类型----->  {}",resolvableType.getRawClass().getName());
        }
        return assignableFrom;
    }

    private boolean isAssignableFrom(ResolvableType type, Class<?>... supportedTypes) {
        if (type != null) {
            for (Class<?> supportedType : supportedTypes) {
                if (supportedType.isAssignableFrom(type.getRawClass())) {
                    return true;
                }
            }
        }
        return false;
    }

}
