package top.doudou.base.properties;

import io.swagger.annotations.ApiModelProperty;
import lombok.Data;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Configuration;

import javax.annotation.PostConstruct;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * @Description 线程配置
 * @Author 傻男人 <244191347@qq.com>
 * @Date 2020-10-23 11:26
 * @Version V1.0
 */
@Data
@Configuration
@ConfigurationProperties(prefix = "executor")
public class ExecutorProperties {

    /**
     * 最大线程数
     */
    @ApiModelProperty("最大线程数")
    private Integer maxPoolSize;

    /**
     * 核心线程数
     */
    @ApiModelProperty("核心线程数")
    private Integer corePoolSize;

    /**
     * 最大空闲时间
     */
    private Integer maxTimeoutSec = 300;

    /**
     * 是否允许核心线程数超时
     */
    private boolean allowCoreThreadTimeOut = false;

    /**
     * 线程名称的前缀
     */
    private String threadNamePrefix ;

    /**
     * 阻塞队列的最大容量
     */
    @ApiModelProperty("队列最大容量")
    private int capacity = 500;

    /**
     * 线程拒绝的方式
     */
    private RejectEnum rejectHandler = RejectEnum.CallerRunsPolicy;


    @ApiModelProperty("保持活跃秒数")
    private int keepAliveSecond = 10;

    @PostConstruct
    public void init(){
        //获得cpu的核心数量
        int core = Runtime.getRuntime().availableProcessors();

        //配置核心线程数 - cpu核心数 - CPU密集型的应用
        if (corePoolSize == null || corePoolSize <= 0) {
            corePoolSize = core;
        }

        //配置最大线程数 - cpu核心数 * 2 - IO密集型的应用
        if (maxPoolSize == null || maxPoolSize <= 0) {
            maxPoolSize = core * 2;
        }
    }

    @Override
    public String toString() {
        return "executor info：{" +
                "threadNamePrefix=" + threadNamePrefix +
                "maxPoolSize=" + maxPoolSize +
                ", corePoolSize=" + corePoolSize +
                ", capacity=" + capacity +
                ", allowCoreThreadTimeOut=" + allowCoreThreadTimeOut +
                ", maxTimeoutSec=" + maxTimeoutSec +
                ", rejectHandler=" + rejectHandler +
                ", keepAliveSecond=" + keepAliveSecond +
                '}';
    }


    public enum RejectEnum {
        //抛出异常的方式
        AbortPolicy(new ThreadPoolExecutor.AbortPolicy()),
        //提交任务的线程自行执行该任务
        CallerRunsPolicy(new ThreadPoolExecutor.CallerRunsPolicy()),
        //线程池会放弃当前等待队列中，最久的任务，当前被拒绝的任何放入队列
        DiscardOldestPolicy(new ThreadPoolExecutor.DiscardOldestPolicy()),
        //直接丢弃当前拒绝的任务
        DiscardPolicy(new ThreadPoolExecutor.DiscardPolicy());

        /**
         * 拒绝策略的处理器
         */
        private RejectedExecutionHandler rejectedExecutionHandler;

        RejectEnum(RejectedExecutionHandler rejectedExecutionHandler) {
            this.rejectedExecutionHandler = rejectedExecutionHandler;
        }
        public RejectedExecutionHandler getRejectedExecutionHandler() {
            return rejectedExecutionHandler;
        }
    }

}
