package top.doudou.base.util;

import cn.hutool.core.net.Ipv4Util;
import cn.hutool.core.util.StrUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import top.doudou.base.entity.IpPconlineDto;
import top.doudou.core.constant.CommonConstant;
import top.doudou.core.constant.RegexConstant;
import top.doudou.core.convert.ConvertBeanUtils;
import top.doudou.core.exception.CustomException;
import top.doudou.core.exception.ExceptionUtils;
import top.doudou.core.util.FastAssert;
import top.doudou.core.util.ReUtil;

import javax.servlet.http.HttpServletRequest;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.net.*;
import java.nio.charset.Charset;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;


/**
 * @Description 获取ip
 * @Author 傻男人 <244191347@qq.com>
 * @Date 2020-09-28 16:10
 * @Version V1.0
 */
@Slf4j
public class IpHelper {

    private final static String UNKNOWN = "unknown";

    /**
     * 太平洋网络IP地址查询Web接口
     */
    private static final String PCONLINE_URL = "http://whois.pconline.com.cn/ipJson.jsp?json=true";

    private static String serverIp;

    static {
        try {
            serverIp = InetAddress.getLocalHost().getHostAddress();
        } catch (UnknownHostException e) {
            log.error("未知的网络，错误的原因：{}",e.getMessage());
            serverIp = CommonConstant.LOCAL_HOST;
        }
    }

    /**
     * 获取当前机器的IP
     *
     * @return /
     */
    public static String getLocalIp() {
        try {
            InetAddress candidateAddress = null;
            // 遍历所有的网络接口
            for (Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces(); interfaces.hasMoreElements();) {
                NetworkInterface anInterface = interfaces.nextElement();
                // 在所有的接口下再遍历IP
                for (Enumeration<InetAddress> inetAddresses = anInterface.getInetAddresses(); inetAddresses.hasMoreElements();) {
                    InetAddress inetAddr = inetAddresses.nextElement();
                    // 排除loopback类型地址
                    if (!inetAddr.isLoopbackAddress()) {
                        if (inetAddr.isSiteLocalAddress()) {
                            // 如果是site-local地址，就是它了
                            return inetAddr.getHostAddress();
                        } else if (candidateAddress == null) {
                            // site-local类型的地址未被发现，先记录候选地址
                            candidateAddress = inetAddr;
                        }
                    }
                }
            }
            if (candidateAddress != null) {
                return candidateAddress.getHostAddress();
            }
            // 如果没有发现 non-loopback地址.只能用最次选的方案
            InetAddress jdkSuppliedAddress = InetAddress.getLocalHost();
            if (jdkSuppliedAddress == null) {
                return "";
            }
            return jdkSuppliedAddress.getHostAddress();
        } catch (Exception e) {
            return "127.0.0.1";
        }
    }


    /**
     * 获取ip
     *
     * @return {String}
     */
    public static String getRequestIp() {
        return getRequestIp(ServletUtils.getRequest());
    }

    /**
     * 获取IP地址
     * 使用Nginx等反向代理软件， 则不能通过request.getRemoteAddr()获取IP地址
     * 如果使用了多级反向代理的话，X-Forwarded-For的值并不止一个，而是一串IP地址，X-Forwarded-For中第一个非unknown的有效IP字符串，则为真实IP地址
     * @param request HttpServletRequest
     * @return {String}
     */
    public static String getRequestIp(HttpServletRequest request) {
        FastAssert.notNull(request, "HttpServletRequest is null");
        String ipAddress = null;
        try {
            //X-Forwarded-For：Squid 服务代理
            ipAddress = request.getHeader("X-Forwarded-For");
            if (StringUtils.isBlank(ipAddress) || UNKNOWN.equalsIgnoreCase(ipAddress)) {
                //Proxy-Client-IP：apache 服务代理
                ipAddress = request.getHeader("Proxy-Client-IP");
                if(StringUtils.isNotBlank(ipAddress)){
                    log.info("get Proxy-Client-IP  ip value :{}",ipAddress);
                }
            }
            if (StringUtils.isBlank(ipAddress)|| UNKNOWN.equalsIgnoreCase(ipAddress)) {
                //WL-Proxy-Client-IP：weblogic 服务代理
                ipAddress = request.getHeader("WL-Proxy-Client-IP");
                if(StringUtils.isNotBlank(ipAddress)){
                    log.info("get WL-Proxy-Client-IP  ip value :{}",ipAddress);
                }
            }
            if (StringUtils.isBlank(ipAddress)|| UNKNOWN.equalsIgnoreCase(ipAddress)) {
                //HTTP_CLIENT_IP：有些代理服务器
                ipAddress = request.getHeader("HTTP_CLIENT_IP");
                if(StringUtils.isNotBlank(ipAddress)){
                    log.info("get HTTP_CLIENT_IP  ip value :{}",ipAddress);
                }
            }
            if (StringUtils.isBlank(ipAddress)|| UNKNOWN.equalsIgnoreCase(ipAddress)) {
                //X-Real-IP：nginx服务代理
                ipAddress = request.getHeader("X-Real-IP");
                if(StringUtils.isNotBlank(ipAddress)){
                    log.info("get X-Real-IP  ip value :{}",ipAddress);
                }
            }
            if (StringUtils.isBlank(ipAddress)|| UNKNOWN.equalsIgnoreCase(ipAddress)) {
                ipAddress = request.getHeader("HTTP_X_FORWARDED_FOR");
                if(StringUtils.isNotBlank(ipAddress)){
                    log.info("get HTTP_X_FORWARDED_FOR  ip value :{}",ipAddress);
                }
            }
            //有些网络通过多层代理，那么获取到的ip就会有多个，一般都是通过逗号（,）分割开来，并且第一个ip为客户端的真实IP
            if (ipAddress != null && ipAddress.length() != 0) {
                if(StringUtils.isNotBlank(ipAddress)){
                    log.info("get 多层代理  ip value :{}",ipAddress);
                }
                ipAddress = ipAddress.split(",")[0];
            }
            //还是不能获取到，最后再通过request.getRemoteAddr();获取
            if (StringUtils.isBlank(ipAddress) || UNKNOWN.equalsIgnoreCase(ipAddress)) {
                ipAddress = request.getRemoteAddr();
                if(StringUtils.isNotBlank(ipAddress)){
                    log.info("get request.getRemoteAddr()  ip value :{}",ipAddress);
                }
            }
            if (CommonConstant.LOCAL_IP.equals(ipAddress)) {
                ipAddress = getServerIp();
                if(StringUtils.isNotBlank(ipAddress)){
                    log.info("get server ip value :{}",ipAddress);
                }
            }
        } catch (Exception e) {
            log.error("获取ip异常,返回结果为空,原因为:{}",e.getMessage());
            ipAddress = "";
        }
        return ipAddress;
    }

    /**
     * 获取服务器的ip地址
     *
     * @return String
     */
    public static String getServerIp() {
        return serverIp;
    }

    /**
     * 判断ip格式是否正确
     * @param ipStr
     * @return
     */
    public static boolean check(String ipStr){
        return ReUtil.isMatch(RegexConstant.IPV4_PATTERN,ipStr);
    }

    /**
     * 获取所有的网卡的ipv4地址,key为网卡地址，value为ip地址
     *
     * @return hash map
     */
    public static Map<String, String> getLocalIPV4() {
        Map<String, String> map = new HashMap<>();
        InetAddress ip = null;
        try {
            Enumeration<NetworkInterface> netInterfaces = NetworkInterface.getNetworkInterfaces();
            while (netInterfaces.hasMoreElements()) {
                NetworkInterface ni = netInterfaces.nextElement();
                Enumeration<InetAddress> ips = ni.getInetAddresses();
                while (ips.hasMoreElements()) {
                    ip = ips.nextElement();
                    if (ip instanceof Inet4Address) {
                        map.put(ni.getName(), ip.getHostAddress());
                    }
                }
            }
        } catch (Exception e) {
            log.error("获取ipv4错误");
            log.error(ExceptionUtils.toString(e));
        }
        return map;
    }

    /**
     * 获取所有ipv6地址
     *
     * @return hash map
     */
    public static Map<String, String> getLocalIPV6() {
        Map<String, String> map = new HashMap<>();
        InetAddress ip = null;
        try {
            Enumeration<NetworkInterface> netInterfaces = NetworkInterface.getNetworkInterfaces();
            while (netInterfaces.hasMoreElements()) {
                NetworkInterface ni = netInterfaces.nextElement();
                Enumeration<InetAddress> ips = ni.getInetAddresses();
                while (ips.hasMoreElements()) {
                    ip = ips.nextElement();
                    if (ip instanceof Inet6Address) {
                        map.put(ni.getName(), ip.getHostAddress());
                    }
                }
            }
        } catch (Exception e) {
            log.error("获取ipv6错误");
            log.error(ExceptionUtils.toString(e));
        }
        return map;
    }

    /**
     * 把IP的long值转换成字符串 v4地址：xx.xx.xx.xx
     *
     * @param longIP IP的long表示形式
     * @return IP地址
     */
    public static String longToIp(long longIP) {
        final StringBuilder sb = StrUtil.builder();
        // 直接右移24位
        sb.append((longIP >>> 24)).append(".");
        // 将高8位置0，然后右移16位
        sb.append(((longIP & 0x00FFFFFF) >>> 16)).append(".");
        sb.append(((longIP & 0x0000FFFF) >>> 8)).append(".");
        sb.append((longIP & 0x000000FF));
        return sb.toString();
    }


    /**
     * 根据ip地址(xxx.xxx.xxx.xxx)计算出long型的数据
     *
     * @param ipStr 字符串IP
     * @return IP对应的long值
     */
    public static long ipToLong(String ipStr) {
        if(!check(ipStr)){
            throw new CustomException("{}  not ip type",ipStr);
        }
        String[] ip = ipStr.split("\\.");
        return (Long.valueOf(ip[0]) << 24) + (Long.valueOf(ip[1]) << 16)
                + (Long.valueOf(ip[2]) << 8) + Long.valueOf(ip[3]);
    }

    /**
     * 判断ip是否属于某个IP范围段
     *
     * @param ipStr 字符串IP
     * @return ipRange xx.xx.xx.1/10  ip范围后面使用/表示范围
     */
    public static boolean contain(String ipStr,String ipRange) {
        if(!check(ipStr)){
            throw new CustomException("{}  not ip type",ipStr);
        }
        String[] split = ipRange.split("\\.");
        if(split.length != 4){
            log.debug("ip范围的格式不对");
            return false;
        }
        int i = split[3].indexOf("/");
        if(i == -1){
            log.debug("ip范围不是范围的格式  直接匹配字符串");
            return ipStr.equals(ipRange);
        }
        String[] split1 = split[3].split("/");
        String frontIp = split[0] + "." + split[1] + "." + split[2] + "." + split1[0];
        String afterIp = split[0] + "." + split[1] + "." + split[2] + "." + split1[1];
        long ipStrLong = ipToLong(ipStr);
        if(ipStrLong >= ipToLong(frontIp) && ipStrLong <= ipToLong(afterIp)){
            return true;
        }
        if(ipStrLong >= ipToLong(afterIp) && ipStrLong <= ipToLong(frontIp)){
            return true;
        }
        return false;
    }


    public static void main(String[] args) {
//        System.out.println(contain("192.168.1.121", "192.168.1.130/122"));
        System.out.println(getLocalIp());
    }

    /**
     * 调用太平洋网络IP地址查询Web接口（http://whois.pconline.com.cn/），返回ip、地理位置
     * @param ip
     * @return
     */
    public static IpPconlineDto getIpInfo(String ip){
        if(CommonConstant.LOCAL_HOST.equals(ip)){
            return null;
        }
        String url = PCONLINE_URL;
        //查指定ip
        if(!StringUtils.isEmpty(ip)){
            url = PCONLINE_URL+"&ip=" + ip;
        }
        StringBuilder inputLine = new StringBuilder();
        String read;
        try {
            HttpURLConnection urlConnection = (HttpURLConnection) new URL(url).openConnection();
            urlConnection.setRequestProperty("Charset", "GBK");
            BufferedReader in = new BufferedReader(new InputStreamReader(urlConnection.getInputStream(), "GBK"));
            while ((read = in.readLine()) != null) {
                inputLine.append(read);
            }
            in.close();
        } catch (Exception e) {
            throw new CustomException("调用远程获取IP的详细信息失败",e);
        }
        return ConvertBeanUtils.convert(inputLine.toString(),IpPconlineDto.class,Charset.forName("UTF-8"));
    }

    /*
        终极大法：java获取不了，就用js来获取
        <!-- js获取客户ip -->
        <script src="http://whois.pconline.com.cn/ipJson.jsp"></script>
     */
    /**
     * 直接根据访问者的Request，返回ip、地理位置
     * @param request
     * @return
     */
    public static IpPconlineDto getIpInfoByRequest(HttpServletRequest request){
        return getIpInfo(getRequestIp(request));
    }

}
