package top.doudou.common.quartz.listener;

import cn.hutool.core.date.SystemClock;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.event.EventListener;
import org.springframework.scheduling.annotation.Async;
import org.springframework.scheduling.annotation.EnableAsync;
import org.springframework.stereotype.Component;
import top.doudou.base.convert.ConvertBeanUtils;
import top.doudou.common.quartz.entity.dto.sysschedulejob.SysScheduleJobDto;
import top.doudou.common.quartz.entity.dto.sysschedulejoblog.SysScheduleJobLogAddDto;
import top.doudou.common.quartz.event.ScheduleJobEvent;
import top.doudou.common.quartz.service.SysScheduleJobLogService;
import top.doudou.common.quartz.service.SysScheduleJobService;
import top.doudou.common.quartz.util.SpringBeanTaskUtil;

/**
 * @author  傻男人<244191347@qq.com>
 * @Date: 2021-01-27
 * @Version: 1.0
 * @Description: 异步监听定时任务事件，解决job线程无故丢失的问题
 */
@Slf4j
@Component
@EnableAsync
@AllArgsConstructor
public class SysJobListener {

	private final SysScheduleJobLogService sysScheduleJobLogService;

	private final SysScheduleJobService sysScheduleJobService;

	@Async
	@EventListener(ScheduleJobEvent.class)
	public void scheduleJobEventListener(ScheduleJobEvent event) {
		SysScheduleJobDto temp = event.getScheduleJob();
		SysScheduleJobDto scheduleJob = sysScheduleJobService.findByPrimaryId(temp.getId());
		//数据库保存执行记录
		SysScheduleJobLogAddDto jobLog = new SysScheduleJobLogAddDto();
		jobLog.setJobId(scheduleJob.getId());
		jobLog.setBean(scheduleJob.getBean());
		jobLog.setMethod(scheduleJob.getMethod());
		jobLog.setParams(scheduleJob.getParams());

		//任务开始时间
		long startTime = SystemClock.now();
		StringBuilder sb = new StringBuilder();
		//执行任务
		sb.append("任务准备执行，任务ID：" + scheduleJob.getId())
				.append("  bean："+scheduleJob.getBean())
				.append("  method："+scheduleJob.getMethod());
		try {
			SpringBeanTaskUtil.invokeMethod(ConvertBeanUtils.copyProperties(scheduleJob, SysScheduleJobDto.class));
			jobLog.setStatus(1);
			sb.append("  执行成功");
		} catch (Exception e) {
			jobLog.setStatus(0);
			sb.append("  失败的信息为："+e.getMessage());
			jobLog.setError(e.getMessage());
		}finally {
			long times = SystemClock.now() - startTime;
			jobLog.setTime((int)times);
			sb.append("  总耗时：").append(times).append("毫秒");
			log.info(sb.toString());
			sysScheduleJobLogService.save(jobLog);
		}
	}
}
