package top.doudou.common.quartz.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.quartz.CronTrigger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import top.doudou.base.convert.ConvertBeanUtils;
import top.doudou.base.exception.CustomException;
import top.doudou.base.page.PageInfo;
import top.doudou.base.util.FastAssert;
import top.doudou.common.quartz.config.ScheduleManager;
import top.doudou.common.quartz.dao.SysScheduleJobDao;
import top.doudou.common.quartz.entity.SysScheduleJob;
import top.doudou.common.quartz.entity.dto.sysschedulejob.SysScheduleJobAddDto;
import top.doudou.common.quartz.entity.dto.sysschedulejob.SysScheduleJobDto;
import top.doudou.common.quartz.entity.dto.sysschedulejob.SysScheduleJobUpdateDto;
import top.doudou.common.quartz.enums.ScheduleStatus;
import top.doudou.common.quartz.service.SysScheduleJobService;
import top.doudou.common.tool.client.SuggestiveLanguage;
import top.doudou.base.builder.JsonBuilder;
import top.doudou.common.tool.utils.IPageUtil;
import top.doudou.common.tool.utils.ListUtils;

import javax.annotation.PostConstruct;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * @author 傻男人<244191347@qq.com>
 * @Date: 2021-01-26
 * @Version: 1.0
 * @Description: 定时任务详情信息 服务实现类
 */
@Slf4j
@Service
public class SysScheduleJobServiceImpl extends ServiceImpl<SysScheduleJobDao, SysScheduleJob> implements SysScheduleJobService {

    @Autowired
    private ScheduleManager scheduleManager;

    /**
     * 项目启动时，初始化定时器
     */
    @PostConstruct
    public void init(){
        log.info("----初始化定时器----");
        List<SysScheduleJob> list = list();
        log.info("----定时器的个数为：{}----",list.size());
        List<SysScheduleJobDto> result = convertJobList(list);
        result.forEach(item -> {
            CronTrigger trigger = scheduleManager.getCronTrigger(item);
            // 如果定时任务不存在，则创建定时任务
            if (trigger == null) {
                scheduleManager.createScheduleJob(item);
            } else if (ScheduleStatus.NORMAL.getType().equals(item.getStatus())) {
                scheduleManager.resumeJob(item);
            } else if (ScheduleStatus.PAUSE.getType().equals(item.getStatus())) {
                scheduleManager.pauseJob(item);
            }
        });
    }

    /**
     * 添加定时任务详情信息
     *
     * @param sysScheduleJobAddDto
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public SysScheduleJobDto save(SysScheduleJobAddDto sysScheduleJobAddDto) {
        checkBeanAndParams(null,sysScheduleJobAddDto.getBean(),sysScheduleJobAddDto.getParams());
        Integer status = sysScheduleJobAddDto.getStatus();
        if(null != status && status.equals(0)){
            return saveAndStart(sysScheduleJobAddDto);
        }
        SysScheduleJob sysScheduleJob = ConvertBeanUtils.copyProperties(sysScheduleJobAddDto, SysScheduleJob.class);
        sysScheduleJob.setParams(JsonBuilder.gson().toJson(sysScheduleJobAddDto.getParams()));
        this.baseMapper.insert(sysScheduleJob);
        return ConvertBeanUtils.copyProperties(sysScheduleJob, SysScheduleJobDto.class);
    }

    /**
     * 保存并开始定时任务
     *
     * @param scheduleJob
     */
    @Override
    public SysScheduleJobDto saveAndStart(SysScheduleJobAddDto scheduleJob) {
        checkBeanAndParams(null,scheduleJob.getBean(),scheduleJob.getParams());
        SysScheduleJob sysScheduleJob = ConvertBeanUtils.copyProperties(scheduleJob, SysScheduleJob.class);
        sysScheduleJob.setParams(JsonBuilder.gson().toJson(scheduleJob.getParams()));
        this.baseMapper.insert(sysScheduleJob);
        SysScheduleJobDto sysScheduleJobDto = ConvertBeanUtils.copyProperties(sysScheduleJob, SysScheduleJobDto.class);
        sysScheduleJobDto.setParams(JsonBuilder.gson().toJson(scheduleJob.getParams()));
        scheduleManager.createScheduleJob(sysScheduleJobDto);
        return sysScheduleJobDto;
    }

    /**
     * 批量添加定时任务详情信息
     *
     * @param list 批量的list
     * @return
     */
    @Override
    public boolean batchSave(List<SysScheduleJobAddDto> list) {
        List<SysScheduleJob> temp = ListUtils.copyList(list, SysScheduleJob.class);
        return this.saveBatch(temp);
    }

    public void checkBeanAndParams(Long primaryId, String bean,Map<String,Object> params){
        String json = toJson(params);
        SysScheduleJobDto sysScheduleJobDto = findByBeanNameAndParams(bean,json);
        if(null == sysScheduleJobDto){
            return;
        }
        if(null != primaryId && primaryId.equals(sysScheduleJobDto.getId())){
            return;
        }
        throw new CustomException("bean:{} params:{}  已经存在定时任务",bean,params);
    }

    /**
     * 根据beanName 查询定时任务
     * @param bean
     * @return
     */
    public SysScheduleJobDto findByBeanNameAndParams(String bean,String params) {
        if(StringUtils.isBlank(bean)){
            return null;
        }
        QueryWrapper<SysScheduleJob> wrapper = Wrappers.query();
        wrapper.eq(SysScheduleJob.Meta.bean, bean);
        if(StringUtils.isNotBlank(params)){
            wrapper.eq(SysScheduleJob.Meta.params, params);
        }
        wrapper.eq(SysScheduleJob.Meta.delFlag, 0);
        wrapper.last("limit 1");
        SysScheduleJob sysScheduleJob = this.baseMapper.selectOne(wrapper);
        SysScheduleJobDto sysScheduleJobDto = ConvertBeanUtils.copyProperties(sysScheduleJob, SysScheduleJobDto.class);
        return sysScheduleJobDto;
    }


    /**
     * 根据主键id批量逻辑删除定时任务详情信息--逻辑删除
     *
     * @param primaryIds
     * @return 删除的条数
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Integer logicDelete(String primaryIds) {
        if (StringUtils.isBlank(primaryIds)) {
            return 0;
        }
        List<Long> listId = ListUtils.stringToList(primaryIds);
        UpdateWrapper<SysScheduleJob> wrapper = Wrappers.update();
        wrapper.in(SysScheduleJob.Meta.id, listId);

        SysScheduleJob sysScheduleJob = new SysScheduleJob();
        sysScheduleJob.setDelFlag(1);
        sysScheduleJob.setUpdateDate(new Date());

        this.findByPrimaryIds(primaryIds).forEach(item->scheduleManager.deleteScheduleJob(item));
        Integer update = this.baseMapper.update(sysScheduleJob, wrapper);
        return update;
    }

    /**
     * 根据主键id批量逻辑删除定时任务详情信息--物理删除
     *
     * @param primaryIds
     * @return 删除的条数
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Integer directDelete(String primaryIds) {
        if (StringUtils.isBlank(primaryIds)) {
            return 0;
        }
        List<Long> listId = ListUtils.stringToList(primaryIds);
        this.findByPrimaryIds(primaryIds).forEach(item->scheduleManager.deleteScheduleJob(item));
        return this.baseMapper.deleteBatchIds(listId);
    }

    /**
     * 根据id修改定时任务详情信息
     *
     * @param sysScheduleJobUpdateDto
     * @return 修改的条数
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Integer updateById(SysScheduleJobUpdateDto sysScheduleJobUpdateDto) {
        checkBeanAndParams(sysScheduleJobUpdateDto.getId(),sysScheduleJobUpdateDto.getBean(),sysScheduleJobUpdateDto.getParams());
        SysScheduleJob sysScheduleJob = this.baseMapper.selectById(sysScheduleJobUpdateDto.getId());
        if (null == sysScheduleJob) {
            log.info("修改的数据不存在,主键为：{}",sysScheduleJobUpdateDto.getId());
            return null;
        }
        sysScheduleJob = ConvertBeanUtils.copyProperties(sysScheduleJobUpdateDto, SysScheduleJob.class);
        SysScheduleJobDto sysScheduleJobDto = ConvertBeanUtils.copyProperties(sysScheduleJobUpdateDto, SysScheduleJobDto.class);
        scheduleManager.updateScheduleJob(sysScheduleJobDto);
        return this.baseMapper.updateById(sysScheduleJob);
    }

    /**
     * 根据ids查询定时任务详情信息
     *
     * @param primaryIds
     * @return
     */
    @Override
    public List<SysScheduleJobDto> findByPrimaryIds(String primaryIds) {
        List<Long> listIds = ListUtils.stringToList(primaryIds);
        QueryWrapper<SysScheduleJob> wrapper = Wrappers.query();
        wrapper.in(SysScheduleJob.Meta.id, listIds);
        wrapper.eq(SysScheduleJob.Meta.delFlag, 0);
        List<SysScheduleJob> sysScheduleJobs = this.baseMapper.selectList(wrapper);
        return convertJobList(sysScheduleJobs);
    }

    /**
     * 根据id查询定时任务详情信息
     *
     * @param primaryId
     * @return 返回主键为primaryId的实体，如果为空则为null
     */
    @Override
    public SysScheduleJobDto findByPrimaryId(Long primaryId) {
        FastAssert.notEmpty(primaryId, SuggestiveLanguage.notEmpty("主键"));
        SysScheduleJob sysScheduleJob = this.baseMapper.selectById(primaryId);
        if (sysScheduleJob == null || sysScheduleJob.getDelFlag() != 0) {
            return null;
        }
        SysScheduleJobDto sysScheduleJobDto = ConvertBeanUtils.copyProperties(sysScheduleJob, SysScheduleJobDto.class);
        return sysScheduleJobDto;
    }

    /**
     * 分页查询定时任务详情信息
     *
     * @param pageInfo
     * @return
     */
    @Override
    public IPage<SysScheduleJobDto> findByPageHelper(PageInfo pageInfo) {
        Page page = new Page(pageInfo.getPageIndex(), pageInfo.getPageSize());
        IPage<SysScheduleJob> result = this.baseMapper.findByPageHelper(page);
        IPage<SysScheduleJobDto> list = IPageUtil.create(result, SysScheduleJobDto.class);
        return list;
    }

    /**
     * 立即执行任务定时任务
     *
     * @param primaryIds
     * @return
     */
    @Override
    public boolean run(String primaryIds) {
        List<SysScheduleJobDto> list = findByPrimaryIds(primaryIds);
        if(CollectionUtils.isEmpty(list)){
            return false;
        }
        list.forEach(item->scheduleManager.run(item));
        return true;
    }

    /**
     * 暂停定时任务
     *
     * @param primaryIds 定时任务主键
     * @return
     */
    @Override
    public boolean pause(String primaryIds) {
        List<SysScheduleJobDto> list = findByPrimaryIds(primaryIds);
        if(CollectionUtils.isEmpty(list)){
            return false;
        }
        list.forEach(item->scheduleManager.pauseJob(item));
        return true;
    }

    /**
     * 暂停定时任务
     *
     * @param primaryIds 定时任务主键
     * @return
     */
    @Override
    public boolean resume(String primaryIds) {
        List<SysScheduleJobDto> list = findByPrimaryIds(primaryIds);
        if(CollectionUtils.isEmpty(list)){
            return false;
        }
        list.forEach(item->scheduleManager.resumeJob(item));
        return true;
    }

    /**
     * 停止定时任务
     *
     * @param primaryIds 定时任务主键
     * @return
     */
    @Override
    public boolean stop(String primaryIds) {
        return false;
    }

    /**
     * 重启数据库中所有的Job
     *
     * @return 重启的个数
     */
    @Override
    public Integer reStartAllJobs() {
        AtomicInteger ai = new AtomicInteger(0);
        synchronized (log) {
            List<SysScheduleJobDto> result = findEffectiveJob();
            log.info("----定时器的个数为：{}----",result.size());
            result.forEach(item -> {
                CronTrigger trigger = scheduleManager.getCronTrigger(item);
                // 如果定时任务不存在，则创建定时任务
                if (trigger == null) {
                    scheduleManager.createScheduleJob(item);
                } else if (ScheduleStatus.NORMAL.getType().equals(item.getStatus())) {
                    log.info("任务 : {} , Because {} status is {}", item.getRemark(), item.getBean(), item.getStatus());
                } else {
                    ai.incrementAndGet();
                    scheduleManager.resumeJob(item);
                }
            });
        }
        return ai.get();
    }

    /**
     * 查询系统未删除的定时任务
     * @return
     */
    public List<SysScheduleJobDto> findEffectiveJob(){
        QueryWrapper<SysScheduleJob> wrapper = Wrappers.query();
        wrapper.eq(SysScheduleJob.Meta.delFlag,0);
        List<SysScheduleJob> sysScheduleJobs = this.baseMapper.selectList(wrapper);
        return convertJobList(sysScheduleJobs);
    }
    private String toJson(Map<String,Object> map){
        if(MapUtils.isEmpty(map)){
            return null;
        }
        return JsonBuilder.gson().toJson(map);
    }

    private List<SysScheduleJobDto> convertJobList(List<SysScheduleJob> list){
        List<SysScheduleJobDto> result = Lists.newArrayList();
        list.forEach(item->{
            SysScheduleJobDto sysScheduleJobDto = ConvertBeanUtils.copyProperties(item, SysScheduleJobDto.class);
            result.add(sysScheduleJobDto);
        });
        return result;
    }

}
