package top.doudou.common.redis.lock;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.core.StringRedisTemplate;
import top.doudou.core.random.RandomUtils;

import java.util.concurrent.TimeUnit;

/**
 * @Description
 * @author 傻男人<244191347@qq.com>
 * @Date 2020-11-20 17:41
 * @Version V1.0
 */
@Configuration
public class RedisLockImpl implements RedisLock {

    @Autowired
    private StringRedisTemplate stringRedisTemplate;

    private ThreadLocal<String> threadLocal = new ThreadLocal<>();

    /**
     * 重入的次数
     */
    private ThreadLocal<Integer> reentryCount = new ThreadLocal<>();

    /**
     * 尝试加锁
     * @param key 加锁的key
     * @param timeout  超时的时间
     * @param unit 时间的单位
     * @return
     */
    @Override
    public boolean tryLock(String key, long timeout, TimeUnit unit) {
        Boolean isLocked = false;
        if (threadLocal.get() == null) {
            String uuid = RandomUtils.randomUUID(32);
            threadLocal.set(uuid);
            isLocked = stringRedisTemplate.opsForValue().setIfAbsent(key, uuid, timeout, unit);
            // 尝试获取锁失败，则自旋获取锁直至成功
            if (!isLocked) {
                for (;;) {
                    isLocked = stringRedisTemplate.opsForValue().setIfAbsent(key, uuid, timeout, unit);
                    if (isLocked) {
                        break;
                    }
                }
            }
            // 启动新的线程来定期检查当前线程是否执行完成，并更新过期时间
            new Thread(new UpdateLockTimeoutTask(uuid, stringRedisTemplate, key)).start();
        } else {
            isLocked = true;
        }
        // 重入次数加1
        if (isLocked) {
            Integer count = reentryCount.get() == null ? 0 : reentryCount.get();
            reentryCount.set(count++);
        }

        return isLocked;
    }

    /**
     * 解锁操作
     * @param key 加锁的key
     */
    @Override
    public void releaseLock(String key) {
        // 判断当前线程所对应的uuid是否与Redis对应的uuid相同，再执行删除锁操作
        String uuid = stringRedisTemplate.opsForValue().get(key);
        Integer count = reentryCount.get();
        if (threadLocal.get().equals(uuid)) {
            // 计数器减为0时才能释放锁
            if (count == null || --count <= 0) {
                stringRedisTemplate.delete(key);
                // 获取更新锁超时时间的线程并中断
                String threadId = stringRedisTemplate.opsForValue().get(uuid);
                Thread updateLockTimeoutThread = ThreadUtils.getThreadByThreadId(threadId);
                if (updateLockTimeoutThread != null) {
                    // 中断更新锁超时时间的线程
                    updateLockTimeoutThread.interrupt();
                    stringRedisTemplate.delete(uuid);
                }
            }
            if(null == count || 0 == count){
                reentryCount.remove();
                threadLocal.remove();
            }
        }
    }
}