package top.doudou.common.cloud;

import com.aliyun.oss.OSSClient;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import top.doudou.base.exception.CustomException;

import java.io.ByteArrayInputStream;
import java.io.InputStream;

/**
 * @Description  阿里云存储
 * @Author 傻男人 <244191347@qq.com>
 * @Date 2020-11-03 14:51
 * @Version V1.0
 */
@Slf4j
public class AliyunCloudStorageService extends AbstractCloudStorageService {

    public AliyunCloudStorageService(CloudStorageProperties config){
        this.config = config;
    }

    @Override
    public String upload(byte[] data, String path) {
        return upload(new ByteArrayInputStream(data), path);
    }

    @Override
    public String upload(InputStream inputStream, String path) {
        OSSClient client = new OSSClient(config.getEndPoint(), config.getAccessKey(),
                config.getSecretKey());
        try {
            client.putObject(config.getBucketName(), path, inputStream);
            client.shutdown();
        } catch (Exception e){
            log.error("上传文件到阿里云oss失败");
            throw new CustomException(e);
        }
        return String.format("%s/%s", config.getDomain(), path);
    }

    @Override
    public String uploadSuffix(byte[] data, String suffix) {
        return upload(data, getPath(config.getPrefix(), suffix));
    }

    /**
     * 文件上传
     *
     * @param data   文件字节数组
     * @param prefix  路径前缀
     * @param suffix 后缀(一般为文件的类型比如jgp)
     * @return 返回http地址
     */
    @Override
    public String uploadSuffix(byte[] data, String prefix, String suffix) {
        return uploadSuffix(new ByteArrayInputStream(data),prefix,suffix);
    }


    @Override
    public String uploadSuffix(InputStream inputStream, String suffix) {
        return upload(inputStream, getPath(config.getPrefix(), suffix));
    }

    /**
     * 文件上传
     *
     * @param inputStream 字节流
     * @param prefix      路径前缀
     * @param suffix      后缀(一般为文件的类型比如jgp)
     * @return 返回http地址
     */
    @Override
    public String uploadSuffix(InputStream inputStream, String prefix, String suffix) {
        String configPrefix = config.getPrefix();
        if(StringUtils.isBlank(configPrefix)){
            return upload(inputStream, getPath(prefix, suffix));
        }
        if(prefix.startsWith("/")){
            return upload(inputStream, getPath(configPrefix+prefix, suffix));
        }
        if(prefix.startsWith("\\")){
            return upload(inputStream, getPath(configPrefix+"/"+prefix.substring(1), suffix));
        }
        return upload(inputStream, getPath(configPrefix+"/"+prefix, suffix));
    }

}