package top.doudou.common.cloud;

import com.qcloud.cos.COSClient;
import com.qcloud.cos.ClientConfig;
import com.qcloud.cos.auth.BasicCOSCredentials;
import com.qcloud.cos.auth.COSCredentials;
import com.qcloud.cos.model.ObjectMetadata;
import com.qcloud.cos.model.PutObjectRequest;
import com.qcloud.cos.model.PutObjectResult;
import com.qcloud.cos.region.Region;
import org.apache.commons.lang3.StringUtils;
import top.doudou.base.exception.CustomException;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * 腾讯云存储
 *
 * @Author 傻男人 <244191347@qq.com>
 * @Date 2020-11-01 14:21
 * @Version V1.0
 */
public class QcloudCloudStorageService extends AbstractCloudStorageService {
    private COSCredentials credentials;
    private ClientConfig clientConfig;

    public QcloudCloudStorageService(CloudStorageProperties config){
        this.config = config;

        //初始化
        init();
    }

    private void init(){
        //1、初始化用户身份信息(secretId, secretKey)
        credentials = new BasicCOSCredentials(config.getAccessKey(), config.getSecretKey());

    	//2、设置bucket的区域, COS地域的简称请参照 https://cloud.tencent.com/document/product/436/6224
        clientConfig = new ClientConfig(new Region(config.getRegion()));
    }

    @Override
    public String upload(byte[] data, String path) {
        return upload(new ByteArrayInputStream(data), path);
    }

    @Override
    public String upload(InputStream inputStream, String path) {
    	try {
            COSClient client = new COSClient(credentials, clientConfig);

            ObjectMetadata metadata = new ObjectMetadata();
            metadata.setContentLength(inputStream.available());
            String bucketName = config.getBucketName() +"-"+ config.getAppId();
            PutObjectRequest request = new PutObjectRequest(bucketName, path, inputStream, metadata);
            PutObjectResult result = client.putObject(request);

            client.shutdown();
            if(result.getETag() == null){
                throw new CustomException(result.getDateStr());
            }
        } catch (IOException e) {
            throw new CustomException("上传到腾讯云失败");
        }
        return String.format("%s/%s", config.getDomain(), path);
    }

    @Override
    public String uploadSuffix(byte[] data, String suffix) {
        return upload(data, getPath(config.getPrefix(), suffix));
    }

    /**
     * 文件上传
     *
     * @param data   文件字节数组
     * @param prefix 路径前缀
     * @param suffix 后缀(一般为文件的类型比如jgp)
     * @return 返回http地址
     */
    @Override
    public String uploadSuffix(byte[] data, String prefix, String suffix) {
        return uploadSuffix(new ByteArrayInputStream(data),prefix,suffix);
    }

    @Override
    public String uploadSuffix(InputStream inputStream, String suffix) {
        return upload(inputStream, getPath(config.getPrefix(), suffix));
    }

    /**
     * 文件上传
     *
     * @param inputStream 字节流
     * @param prefix      路径前缀
     * @param suffix      后缀(一般为文件的类型比如jgp)
     * @return 返回http地址
     */
    @Override
    public String uploadSuffix(InputStream inputStream, String prefix, String suffix) {
        String configPrefix = config.getPrefix();
        if(StringUtils.isBlank(configPrefix)){
            return upload(inputStream, getPath(prefix, suffix));
        }
        if(prefix.startsWith("/")){
            return upload(inputStream, getPath(configPrefix+prefix, suffix));
        }
        if(prefix.startsWith("\\")){
            return upload(inputStream, getPath(configPrefix+"/"+prefix.substring(1), suffix));
        }
        return upload(inputStream, getPath(configPrefix+"/"+prefix, suffix));
    }
}