package top.doudou.common.cloud;

import com.qiniu.common.Zone;
import com.qiniu.http.Response;
import com.qiniu.storage.Configuration;
import com.qiniu.storage.UploadManager;
import com.qiniu.util.Auth;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import top.doudou.base.exception.CustomException;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * 七牛云存储
 *
 * @Author 傻男人 <244191347@qq.com>
 * @Date 2020-11-01 14:21
 * @Version V1.0
 */
@Slf4j
public class QiniuCloudStorageService extends AbstractCloudStorageService {
    private UploadManager uploadManager;
    private String token;

    public QiniuCloudStorageService(CloudStorageProperties config){
        this.config = config;
        //初始化
        init();
    }

    private void init(){
        uploadManager = new UploadManager(new Configuration(Zone.autoZone()));
        token = Auth.create(config.getAccessKey(), config.getSecretKey()).
                uploadToken(config.getBucketName());

    }

    @Override
    public String upload(byte[] data, String path) {
        try {
            Response res = uploadManager.put(data, path, token);
            if (!res.isOK()) {
                log.error("上传文件到七牛云失败");
                throw new CustomException(res.toString());
            }
        } catch (Exception e) {
            log.error("上传文件到七牛云失败");
            throw new CustomException(e);
        }
        return String.format("%s/%s", config.getDomain(), path);
    }

    @Override
    public String upload(InputStream inputStream, String path) {
        try {
            byte[] data = IOUtils.toByteArray(inputStream);
            return this.upload(data, path);
        } catch (IOException e) {
            log.error("上传文件到七牛云失败");
            throw new CustomException(e);
        }
    }

    @Override
    public String uploadSuffix(byte[] data, String suffix) {
        return upload(data, getPath(config.getPrefix(), suffix));
    }

    /**
     * 文件上传
     *
     * @param data   文件字节数组
     * @param prefix 路径前缀
     * @param suffix 后缀(一般为文件的类型比如jgp)
     * @return 返回http地址
     */
    @Override
    public String uploadSuffix(byte[] data, String prefix, String suffix) {
        return uploadSuffix(new ByteArrayInputStream(data),prefix,suffix);
    }

    @Override
    public String uploadSuffix(InputStream inputStream, String suffix) {
        return upload(inputStream, getPath(config.getPrefix(), suffix));
    }

    /**
     * 文件上传
     *
     * @param inputStream 字节流
     * @param prefix      路径前缀
     * @param suffix      后缀(一般为文件的类型比如jgp)
     * @return 返回http地址
     */
    @Override
    public String uploadSuffix(InputStream inputStream, String prefix, String suffix) {
        String configPrefix = config.getPrefix();
        if(StringUtils.isBlank(configPrefix)){
            return upload(inputStream, getPath(prefix, suffix));
        }
        if(prefix.startsWith("/")){
            return upload(inputStream, getPath(configPrefix+prefix, suffix));
        }
        if(prefix.startsWith("\\")){
            return upload(inputStream, getPath(configPrefix+"/"+prefix.substring(1), suffix));
        }
        return upload(inputStream, getPath(configPrefix+"/"+prefix, suffix));
    }
}
