package top.doudou.common.email;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.stereotype.Component;
import org.thymeleaf.TemplateEngine;
import org.thymeleaf.context.Context;
import top.doudou.base.exception.CustomException;

import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;
import java.io.File;
import java.util.Map;
import java.util.regex.Pattern;

/**
 * @Description
 * @Author 傻男人 <244191347@qq.com>
 * @Date 2020-11-03 10:51
 * @Version V1.0
 */
@Slf4j
@Component
@EnableConfigurationProperties(EmailProperties.class)
@ConditionalOnProperty(name = "email.enabled",havingValue = "true",matchIfMissing = false)
public class EmailHandler {

    @Autowired
    private JavaMailSender javaMailSender;

    @Autowired
    private EmailProperties emailProperties;

    @Autowired
    private TemplateEngine templateEngine;

    public final static Pattern PLATE_NUMBER = Pattern.compile("\\w+@\\w+\\.[a-z]+(\\.[a-z]+)?");

    /**
     * 发送普通邮件
     * @param sendEmail
     * @return
     */
    public boolean sendEmail(SendEmailDto sendEmail) {
        checkParam(sendEmail);
        SimpleMailMessage message = new SimpleMailMessage();
        message.setFrom(emailProperties.getUsername());
        message.setTo(sendEmail.getTo());
        message.setSubject(sendEmail.getSubject());
        message.setText(sendEmail.getContent());
        try{
            javaMailSender.send(message);
        }catch (Exception e){
            log.error("发送纯文本的邮件失败");
            throw new CustomException(e);
        }
        return true;
    }

    /**
     * 发送带附件的普通邮件
     * @param sendEmail
     * @return
     * @throws MessagingException
     */
    public boolean sendAttachmentEmail(SendEmailDto sendEmail){
        checkParam(sendEmail);
        try{
            MimeMessageHelper mimeMessageHelper = createMimeMessageHelper(sendEmail);
            mimeMessageHelper.setText(sendEmail.getContent(), true);
            javaMailSender.send(mimeMessageHelper.getMimeMessage());
        }catch (Exception e){
            log.error("发送带附件的普通邮件的邮件失败");
            throw new CustomException(e);
        }
        return true;
    }

    /**
     * 发送的邮件是富文本（附件，图片，html等）
     * @param sendEmail
     * @return
     * @throws MessagingException
     */
    public boolean sendHtmlEmail(SendEmailDto sendEmail) {
        return sendHtmlEmail(sendEmail,true);
    }

    /**
     * 发送的邮件是富文本（附件，图片，html等）
     * @param sendEmail
     * @param isShowHtml 是否解析html
     * @return
     * @throws MessagingException
     */
    public boolean sendHtmlEmail(SendEmailDto sendEmail,boolean isShowHtml){
        checkParam(sendEmail);
        try{
            MimeMessageHelper mimeMessageHelper = createMimeMessageHelper(sendEmail);
            mimeMessageHelper.setText(sendEmail.getContent(), isShowHtml);
            javaMailSender.send(mimeMessageHelper.getMimeMessage());
        }catch (Exception e){
            log.error("发送的邮件是富文本的邮件失败");
            throw new CustomException(e);
        }
        return true;
    }

    /**
     * html模板邮件
     * @param sendEmailDto 发送邮件的参数
     * @throws MessagingException
     */
    public boolean sendThymeleafEmail(SendEmailDto sendEmailDto){
        checkParam(sendEmailDto);
        try{
            MimeMessageHelper mimeMessageHelper = createMimeMessageHelper(sendEmailDto);
            // 利用 Thymeleaf 模板构建 html 文本
            Context ctx = new Context();
            // 给模板的参数的上下文
            ctx.setVariables(sendEmailDto.getModel());
            // 执行模板引擎，执行模板引擎需要传入模板名、上下文对象
            String emailText = templateEngine.process(sendEmailDto.getTemplate(), ctx);
            mimeMessageHelper.setText(emailText, true);
            javaMailSender.send(mimeMessageHelper.getMimeMessage());
        }catch (Exception e){
            log.error("发送html模板邮件的邮件失败");
            throw new CustomException(e);
        }
        return true;
    }

    private MimeMessageHelper createMimeMessageHelper(SendEmailDto sendEmailDto) throws MessagingException {
        MimeMessage mimeMessage = javaMailSender.createMimeMessage();
        MimeMessageHelper mimeMessageHelper = new MimeMessageHelper(mimeMessage, true,"utf-8");
        mimeMessageHelper.setFrom(emailProperties.getUsername());
        mimeMessageHelper.setTo(sendEmailDto.getTo());
        if(null != sendEmailDto.getCc()){
            mimeMessageHelper.setBcc(sendEmailDto.getCc());
        }
        mimeMessageHelper.setSubject(sendEmailDto.getSubject());
        addAttachment(mimeMessageHelper,sendEmailDto.getAnnex());
        return mimeMessageHelper;
    }

    private boolean isEmail(String... email){
        if(null == email || email.length == 0){
            throw new CustomException("邮件地址出错");
        }
        for (int i = 0; i < email.length; i++) {
            if(!PLATE_NUMBER.matcher(email[i]).find()){
                return false;
            }
        }
        return true;
    }

    private void checkParam(SendEmailDto sendEmailDto){
        if(null != sendEmailDto.getTo() && !isEmail(sendEmailDto.getTo())){
            throw new CustomException("收件人邮件地址错误");
        }
        if(null != sendEmailDto.getCc() && !isEmail(sendEmailDto.getCc())){
            throw new CustomException("抄送人邮件地址错误");
        }
    }

    /**
     * 添加附件
     * @param mimeMessageHelper
     * @param attachment
     */
    private void addAttachment(MimeMessageHelper mimeMessageHelper,Map<String, File> attachment){
        if(null != attachment && !attachment.isEmpty()){
            attachment.forEach((key,value)->{
                try {
                    mimeMessageHelper.addAttachment(key,value);
                } catch (MessagingException e) {
                    log.error("邮件添加附件错误");
                    throw new CustomException(e);
                }
            });
        }
    }


}
