package top.doudou.common.verification.code;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.ServletRequestBindingException;
import org.springframework.web.bind.ServletRequestUtils;
import org.springframework.web.context.request.ServletWebRequest;
import top.doudou.common.verification.code.entity.ValidateCodeDo;
import top.doudou.common.verification.code.properties.ValidateCodeProperties;
import top.doudou.core.exception.ValidateCodeException;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Map;

/**
 * @Description 验证码处理器
 * @version: 1.0
 * @Created 傻男人 <244191347@qq.com>
 * @Date 2022-01-20 14:59
 */
@Slf4j
public abstract class AbstractValidateCodeProcessor<C extends ValidateCodeDo> implements ValidateCodeProcessor {

    /**
     * 收集系统中所有的 {@link ValidateCodeGenerator} 接口的实现。
     */
    @Autowired
    private Map<String, ValidateCodeGenerator> validateCodeGenerators;

    @Autowired
    private StorageStrategy<ValidateCodeDo> storageStrategy;

    @Autowired
    private ValidateCodeProperties validateCodeProperties;

    @Autowired
    private HttpServletRequest request;

    @Override
    public void create(HttpServletRequest request, HttpServletResponse response) {
        C validateCode = generate(request);
        save(request, validateCode);
        send(request,response, validateCode);
    }

    @Override
    public void create(ServletWebRequest request) {
        C validateCode = generate(request.getRequest());
        save(request.getRequest(), validateCode);
        send(request.getRequest(),request.getResponse(), validateCode);
    }

    /**
     * 生成校验码
     *
     * @param request
     * @return
     */
    @SuppressWarnings("unchecked")
    private C generate(HttpServletRequest request) {
        String type = getValidateCodeType(request).toString().toLowerCase();
        String generatorName = type + ValidateCodeGenerator.class.getSimpleName();
        ValidateCodeGenerator validateCodeGenerator = validateCodeGenerators.get(generatorName);
        if (validateCodeGenerator == null) {
            throw new ValidateCodeException("验证码生成器" + generatorName + "不存在");
        }
        return (C) validateCodeGenerator.generate(request);
    }

    /**
     * 保存校验码
     *
     * @param request
     * @param validateCode
     */
    private void save(HttpServletRequest request, C validateCode) {
        ValidateCodeType type = getValidateCodeType(request);
        String key = storageStrategy.createKey(request, type);
        log.info("---->  key  :{}",key);
        int expireIn = 0 ;
        if(ValidateCodeType.SMS.equals(type)){
            expireIn = validateCodeProperties.getSms().getExpireIn();
        }else if(ValidateCodeType.IMAGE.equals(type)){
            expireIn = validateCodeProperties.getImage().getExpireIn();
        }
        storageStrategy.save(key,validateCode,expireIn);
    }

    /**
     * 发送校验码，由子类实现
     *
     * @param request
     * @param validateCode
     * @throws Exception
     */
    protected abstract void send(HttpServletRequest request, HttpServletResponse response, C validateCode);

    /**
     * 根据请求的url获取校验码的类型
     *
     * @param request
     * @return
     */
    private ValidateCodeType getValidateCodeType(HttpServletRequest request) {
        String type = StringUtils.substringBefore(getClass().getSimpleName(), "CodeProcessor");
        return ValidateCodeType.valueOf(type.toUpperCase());
    }

    @SuppressWarnings("unchecked")
    @Override
    public void validate(HttpServletRequest request) {

        ValidateCodeType processorType = getValidateCodeType(request);
        String key = storageStrategy.createKey(request, processorType);
        ValidateCodeDo validateCodeDo = storageStrategy.get(key);

        String codeInRequest;
        try {
            codeInRequest = ServletRequestUtils.getStringParameter(request, processorType.getParamNameOnValidate());
        } catch (ServletRequestBindingException e) {
            throw new ValidateCodeException("验证码为空");
        }

        if (StringUtils.isBlank(codeInRequest)) {
            throw new ValidateCodeException("验证码值不能为空");
        }

        if (validateCodeDo == null || !StringUtils.equals(validateCodeDo.getCode(), codeInRequest)) {
            throw new ValidateCodeException("验证码错误或者已过期");
        }
        storageStrategy.delete(key);
    }

    @Override
    public void validate() {
        validate(request);
    }


}
