package top.doudou.web.socket;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Component;
import top.doudou.web.socket.config.FastEndpointConfigure;
import top.doudou.web.socket.entity.MessageDto;
import top.doudou.web.socket.handler.AbsFastWebSocketHandler;

import javax.websocket.*;
import javax.websocket.server.PathParam;
import javax.websocket.server.ServerEndpoint;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;


/**
 * @Description 给指定用户发送消息
 * @Author 傻男人 <244191347@qq.com>
 * @Date 2020-10-14 15:06
 * @Version V1.0
 */
@Slf4j
@Component
@ServerEndpoint(value = "/websocket/notice/{userId}", configurator = FastEndpointConfigure.class)
public class WebSocketNoticeServer extends AbsFastWebSocketHandler {

    private static Map<String, Integer> lengthMap = new ConcurrentHashMap<String, Integer>();

    /**
     * 可能用户同时登录多个设备
     */
    private static Map<Long, List<Session>> userMap = new ConcurrentHashMap<Long, List<Session>>();

    @Override
    @OnOpen
    public void onOpen(@PathParam("userId")Long userId, Session session) {
        open(session);
        cacheUser(userId,session);
    }


    /**
     * 连接关闭调用的方法
     */
    @Override
    @OnClose
    public void onClose(Session session) {
        super.close(session);
        lengthMap.remove(session.getId());
    }

    /**
     * 发生错误时调用
     * 需要实现的类上加@OnError注解
     *
     * @param session
     */
    @Override
    @OnError
    public void onError(Session session, Throwable error) {
        super.error(session,error);
    }

    /**
     * 服务器接收到客户端消息时调用的方法
     * 需要实现的类上加@OnMessage注解
     *
     * @param message 收到的消息
     * @param session
     */
    @Override
    @OnMessage
    public void onMessage(String message, Session session) {
        message(message,session);
    }

    public boolean send(Long userId, MessageDto messageDto){
        List<Session> list = userMap.get(userId);
        if(CollectionUtils.isEmpty(list)){
            return false;
        }
        list.forEach(item->send(item,messageDto));
        return true;
    }

    private synchronized void cacheUser(Long userId,Session session){
        List<Session> list = userMap.get(userId);
        if(CollectionUtils.isEmpty(list)){
            List<Session> temp = new ArrayList<>();
            temp.add(session);
            userMap.put(userId,temp);
            return;
        }
        list.add(session);
        userMap.put(userId,list);
    }

}
