package top.doudou.web.socket.handler;


import lombok.extern.slf4j.Slf4j;
import top.doudou.core.exception.ExceptionUtils;
import top.doudou.web.socket.entity.MessageDto;

import javax.websocket.Session;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.LongAdder;

/**
 * @Description WebSocket的基础父类
 * 实现这个类后，需要再实现后的类上加@ServerEndpoint来指定websocket监听端点与@Component扫描
 * @Author 傻男人 <244191347@qq.com>
 * @Date 2020-10-14 15:06
 * @Version V1.0
 */
@Slf4j
public abstract class AbsFastWebSocketHandler {

    /**
     * 连接集合
     */
    protected static Map<String, Session> sessionMap = new ConcurrentHashMap<String, Session>();

    /**
     * 连接的个数
     */
    protected static LongAdder count = new LongAdder();


    public String websocketType;

    public String getWebsocketType() {
        return websocketType;
    }

    public void setWebsocketType(String websocketType) {
        this.websocketType = websocketType;
    }

    /**
     * 连接建立成功调用的方法
     *需要实现的类上加@OnOpen注解
     * @param session
     */
    public void onOpen(Session session){};

    /**
     * 连接建立成功调用的方法
     * 要实现的类上加@OnOpen注解
     * @param userId 用户id
     * @param session
     */
    public void onOpen(Long userId,Session session){};

    /**
     * 连接建立成功调用的方法
     * 要实现的类上加@OnOpen注解
     * @param uniqueKey 唯一键  参数传入
     * @param session
     */
    public void onOpen(String uniqueKey,Session session){};

    /**
     * 连接关闭调用的方法
     * 需要实现的类上加@OnClose注解
     * @param session
     */
    public abstract void onClose(Session session);

    /**
     * 发生错误时调用
     * 需要实现的类上加@OnError注解
     * @param session
     * @param error
     */
    public abstract void onError(Session session,Throwable error);

    /**
     * 服务器接收到客户端消息时调用的方法
     * 需要实现的类上加@OnMessage注解
     * @param session
     * @param message 收到的消息
     */
    public abstract void onMessage(String message,Session session);

    /**
     * 广播所有
     * @param message
     */
    public void sendAll(MessageDto message){
        sessionMap.forEach((key,value)->{
            send(value,message);
        });
    }

    /**
     * 默认onOpen的实现
     * @param session
     */
    protected void open(Session session) {
        count.increment();
        log.info("websocket的类型为：{}   新的连接：{},当前连接的人数：{}",websocketType,session.getId(),count.intValue());
        sessionMap.put(session.getId(),session);
    }

    /**
     * 默认onClose的实现
     * @param session
     */
    protected void close(Session session) {
        sessionMap.remove(session.getId());
        count.decrement();
        log.info("websocket的类型为：{}   连接关闭了：{},当前在线的人数：{}",websocketType,session.getId(),count.intValue());
    }

    /**
     * 默认onError的实现
     * @param session
     */
    protected void error(Session session, Throwable error) {
        log.info("websocket的类型为：{}   连接发生了错误：{},错误的原因为：{}",websocketType,session.getId(),error.getLocalizedMessage());
        log.error(ExceptionUtils.toString(error));
    }

    /**
     * 默认onMessage的实现
     * @param session
     */
    protected void message(String message, Session session) {
        System.out.println("收到消息："+ message);
    }

    /**
     * 封装一个send方法，发送消息到前端
     */
    protected void send(Session session, MessageDto message) {
        try {
            session.getBasicRemote().sendObject(message);
        } catch (Exception e) {
            log.error(ExceptionUtils.toString(e));
        }
    }

    protected void send(Session session, String message) {
        try {
            session.getBasicRemote().sendText(message);
        } catch (Exception e) {
            log.error(ExceptionUtils.toString(e));
        }
    }
}
