package top.doudou.web.socket;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;
import top.doudou.core.exception.ExceptionUtils;
import top.doudou.core.properties.CustomLogProperties;
import top.doudou.core.util.LogUtil;
import top.doudou.core.util.file.FileReadAndWriteUtil;
import top.doudou.web.socket.config.FastEndpointConfigure;
import top.doudou.web.socket.handler.AbsFastWebSocketHandler;

import javax.websocket.*;
import javax.websocket.server.PathParam;
import javax.websocket.server.ServerEndpoint;
import java.io.File;
import java.util.Arrays;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;


/**
 * @Description WebSocket获取实时日志并输出到Web页面
 * @Author 傻男人 <244191347@qq.com>
 * @Date 2020-10-14 15:06
 * @Version V1.0
 */
@Slf4j
@Component
@ServerEndpoint(value = "/dynamic/{applicationName}", configurator = FastEndpointConfigure.class)
@EnableConfigurationProperties(CustomLogProperties.class)
public class DynamicLogEndpoint extends AbsFastWebSocketHandler {

    @Autowired
    private CustomLogProperties customLogProperties;

    private static Map<String, Integer> lengthMap = new ConcurrentHashMap<String, Integer>();

    /**
     * 连接建立成功调用的方法
     */
    @OnOpen
    public void onOpen(Session session,@PathParam("applicationName") String applicationName) {
        setWebsocketType("其他服务动态日志");
        open(session);
        if(StringUtils.isBlank(applicationName) || "undefined".equals(applicationName)){
            send(session,"请在地址栏传入应用的名字，例如：?name=user");
            return;
        }
        String filePath = customLogProperties.getDefaultLogFile();
        if(!new File(filePath).exists()){
            send(session,"服务名或者文件的路径错误，故无法读取到争取的日志");
            return;
        }
        //默认从第一行开始
        lengthMap.put(session.getId(), 1);
        new Thread(()->{
            boolean first = true;
            while (sessionMap.get(session.getId()) != null) {
                try {
                    Object[] lines = FileReadAndWriteUtil.read(filePath);
                    //只取从上次之后产生的日志
                    Object[] copyOfRange = Arrays.copyOfRange(lines, lengthMap.get(session.getId()), lines.length);
                    //存储最新一行开始
                    lengthMap.replace(session.getId(), lines.length);
                    if(copyOfRange.length > 0){
                        String result = LogUtil.handler(copyOfRange, first);
                        send(session,result);
                    }
                    Thread.sleep(1000);
                } catch (Exception e) {
                    log.error(ExceptionUtils.toString(e));
                }
            }
        }).start();
    }

    /**
     * 连接关闭调用的方法
     */
    @Override
    @OnClose
    public void onClose(Session session) {
        super.close(session);
        lengthMap.remove(session.getId());
    }

    /**
     * 发生错误时调用
     * 需要实现的类上加@OnError注解
     *
     * @param session
     */
    @Override
    @OnError
    public void onError(Session session, Throwable error) {
        super.error(session,error);
    }

    /**
     * 服务器接收到客户端消息时调用的方法
     * 需要实现的类上加@OnMessage注解
     *
     * @param message 收到的消息
     * @param session
     */
    @Override
    @OnMessage
    public void onMessage(String message, Session session) {

    }
}
