package top.doudou.web.socket;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;
import top.doudou.core.exception.ExceptionUtils;
import top.doudou.core.properties.CustomLogProperties;
import top.doudou.core.util.LogUtil;
import top.doudou.core.util.file.FileReadAndWriteUtil;
import top.doudou.web.socket.config.FastEndpointConfigure;
import top.doudou.web.socket.handler.AbsFastWebSocketHandler;

import javax.websocket.*;
import javax.websocket.server.ServerEndpoint;
import java.util.Arrays;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;


/**
 * @Description WebSocket获取实时日志并输出到Web页面
 * @Author 傻男人 <244191347@qq.com>
 * @Date 2020-10-14 15:06
 * @Version V1.0
 */
@Slf4j
@Component
@ServerEndpoint(value = "/websocket/dynamic/log", configurator = FastEndpointConfigure.class)
@EnableConfigurationProperties(CustomLogProperties.class)
public class DynamicLogServer extends AbsFastWebSocketHandler {

    @Value("${spring.application.name}")
    private String applicationName;

    @Autowired
    private CustomLogProperties customLogProperties;

    private static Map<String, Integer> lengthMap = new ConcurrentHashMap<String, Integer>();

    /**
     * 连接建立成功调用的方法
     */
    @Override
    @OnOpen
    public void onOpen(Session session) {
        setWebsocketType("当前服务动态日志");
        open(session);
        //默认从第一行开始
        lengthMap.put(session.getId(), 1);
        new Thread(()->{
            boolean first = true;
            while (sessionMap.get(session.getId()) != null) {
                try {
                    Object[] lines = FileReadAndWriteUtil.read(customLogProperties.getDefaultLogFile());
                    //只取从上次之后产生的日志
                    Object[] copyOfRange = Arrays.copyOfRange(lines, lengthMap.get(session.getId()), lines.length);
                    //存储最新一行开始
                    lengthMap.replace(session.getId(), lines.length);
                    if(copyOfRange.length > 0){
                        String result = LogUtil.handler(copyOfRange, first);
                        send(session,result);
                    }
                    Thread.sleep(1000);
                } catch (Exception e) {
                    log.error(ExceptionUtils.toString(e));
                }
            }
        }).start();
    }

    /**
     * 连接关闭调用的方法
     */
    @Override
    @OnClose
    public void onClose(Session session) {
        super.close(session);
        lengthMap.remove(session.getId());
    }

    /**
     * 发生错误时调用
     * 需要实现的类上加@OnError注解
     *
     * @param session
     */
    @Override
    @OnError
    public void onError(Session session, Throwable error) {
        super.error(session,error);
    }

    /**
     * 服务器接收到客户端消息时调用的方法
     * 需要实现的类上加@OnMessage注解
     *
     * @param message 收到的消息
     * @param session
     */
    @Override
    @OnMessage
    public void onMessage(String message, Session session) {

    }

    public static void main(String[] args) {
        String str = "<br/>2020-11-07 11:22:40&nbsp;&nbsp;&nbsp;[<span style='color: green;'>INFO</span> ]&nbsp;&nbsp;&nbsp;[http-nio-9093-exec-9]&nbsp;&nbsp;&nbsp;<span style='color: black;'>top.doudou.common.tool.aspect.RequestLogBaseAspect</span>&nbsp;&nbsp;&nbsp;[55] &nbsp;&nbsp;&nbsp;<span style='color: green;'> [1bb91801b0294bcad]</span>&nbsp;&nbsp;&nbsp;<span style='color: green;'>========================START========================</span>";
        str.replaceAll(" ","&nbsp;");
    }

}
