package top.bluesword.util.algorithm.totp;

import org.jetbrains.annotations.NotNull;
import top.bluesword.util.exception.SwordRuntimeException;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.math.BigInteger;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;

/**
 * 信息验证码生成工具
 * 基于RFC6238
 * @author 李林峰
 */
public class SwordTimeAuthenticator {

    private static final int[] DIGITS_POWER = {1, 10, 100, 1000, 10000, 100000, 1000000, 10000000, 100000000};

    private SwordTimeAuthenticator() {
    }

    private static byte[] hmacSha(String crypto, byte[] keyBytes, byte[] text) throws NoSuchAlgorithmException {
        Mac mac = Mac.getInstance(crypto);
        try {
            mac.init(new SecretKeySpec(keyBytes, "RAW"));
        } catch (InvalidKeyException e) {
            throw new SwordRuntimeException("密钥生成异常");
        }
        return mac.doFinal(text);
    }

    private static byte[] hexStr2Bytes(String hexStr) {
        byte[] bArray = new BigInteger("10" + hexStr, 16).toByteArray();
        byte[] ret = new byte[bArray.length - 1];
        System.arraycopy(bArray, 1, ret, 0, ret.length);
        return ret;
    }

    /**
     * @param key          数字密钥
     * @param message         携带信息
     * @param codeDigits 生成信息码位数,最大8位
     */
    private static String generateOTP(String key, @NotNull String message, int codeDigits){
        String crypto = "HmacSHA512";
        String result;
        if (message.length() < 16) {
            StringBuilder builder = new StringBuilder();
            for (int i = 0; i < 16 - message.length(); i++) builder.append(0);
            message = builder.append(message).toString();
        }
        byte[] msg = message.getBytes();
        byte[] k = hexStr2Bytes(key);
        byte[] hash;
        try {
            hash = hmacSha(crypto, k, msg);
        } catch (NoSuchAlgorithmException e) {
            throw new SwordRuntimeException("Hmac算法配置异常");
        }
        int offset = hash[hash.length - 1] & 0xf;
        int binary = ((hash[offset] & 0x7f) << 24)
                | ((hash[offset + 1] & 0xff) << 16)
                | ((hash[offset + 2] & 0xff) << 8) | (hash[offset + 3] & 0xff);
        int otp = binary % DIGITS_POWER[codeDigits];
        result = Integer.toString(otp);
        if (result.length() < codeDigits) {
            String format = "%0" + (codeDigits - result.length()) + "d%s";
            result = String.format(format, 0, result);
        }
        return result;
    }

    /**
     * 生成基于时间的令牌
     * @param key 密钥
     * @param message 加密信息
     * @param codeDigits 验证码位数
     * @return 基于时间的令牌
     */
    public static String generateTimeOTP(String key, String message, int codeDigits) {
        return generateOTP(key, System.currentTimeMillis() / (30 * 60 * 1000) + message, codeDigits);
    }

    /**
     * 验证时间令牌
     * @param key 密钥
     * @param message 加密信息
     * @param codeDigits 验证码位数
     * @param code 令牌
     * @return 是否通过验证
     */
    public static boolean checkTimeOTP(String key, String message, int codeDigits, String code) {
        long time = System.currentTimeMillis() / (30 * 60 * 1000);
        return generateOTP(key, time + message, codeDigits).equals(code)
                || generateOTP(key, (time - 1) + message, codeDigits).equals(code);
    }
}
