package top.bluesword.util.network;

import org.apache.commons.codec.CharEncoding;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.NameValuePair;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.fluent.Request;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.entity.StringEntity;
import org.apache.http.message.BasicHeader;
import org.apache.http.message.BasicNameValuePair;
import top.bluesword.util.exception.SwordRuntimeException;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * http请求工具
 * @author 李林峰
 */
public class HttpClientUtil {

    private HttpClientUtil(){}

    /**
     * 发送json格式的post请求
     *
     * @param url  访问地址
     * @param json 请求参数(json字符串)
     * @return 接收的参数
     */
    public static String sendPostJson(String url, String json) {
        return sendPost(url,new StringEntity(json, CharEncoding.UTF_8),
                new Header[]{new BasicHeader("Content-Encoding",CharEncoding.UTF_8),
                        new BasicHeader("content-type","application/json")});
    }

    /**
     * 发送普通格式参数的post请求
     * @param url 请求地址
     * @param paramMap 参数
     * @param headersMap 请求头
     * @return 返回参数
     */
    public static String sendPostParam(String url, Map<String, String> paramMap, Map<String, String> headersMap) {
        try {
            return sendPost(url,new UrlEncodedFormEntity(toPairList(paramMap), CharEncoding.UTF_8),
                    toHeaders(headersMap));
        } catch (IOException e) {
            throw new SwordRuntimeException("发送普通格式参数的post请求失败");
        }
    }

    public static String sendPost(String url, HttpEntity entity, Header[] headers) {
        try {
            return Request.Post(url)
                    .body(entity)
                    .setHeaders(headers)
                    .execute().returnContent().asString();
        } catch (IOException e) {
            throw new SwordRuntimeException("post请求失败");
        }
    }

    public static List<NameValuePair> toPairList(Map<String, String> paramMap) {
        List<NameValuePair> params = new ArrayList<>();
        if (paramMap != null) {
            for (Map.Entry<String, String> entry : paramMap.entrySet()) {
                params.add(new BasicNameValuePair(entry.getKey(), entry.getValue()));
            }
        }
        return params;
    }

    private static Header[] toHeaders(Map<String, String> headersMap) {
        Header[] headers = new Header[headersMap.size()];
        int i = 0;
        for (Map.Entry<String, String> entry : headersMap.entrySet()) {
            headers[i++] = new BasicHeader(entry.getKey(),entry.getValue());
        }
        return headers;
    }

    /**
     * 发送完整信息的get请求
     * @param url 请求地址
     * @param paramMap 参数
     * @param headersMap 请求头
     * @return 返回参数
     */
    public static String sendGet(String url, Map<String, String> paramMap, Map<String, String> headersMap) {

        try {
            String param = "";
            if (paramMap!=null && paramMap.size()>0){
                param = URLEncodedUtils.format(toPairList(paramMap), CharEncoding.UTF_8);
            }
            Request request = Request.Get(url+"?"+param).connectTimeout(5000).socketTimeout(5000);
            if (headersMap!=null){
                for (Map.Entry<String, String> entry : headersMap.entrySet()) {
                    request.addHeader(entry.getKey(),entry.getValue());
                }
            }
            return request.execute().returnContent().asString(Charset.forName(CharEncoding.UTF_8));
        } catch (IOException e) {
            throw new SwordRuntimeException("发送get请求失败");
        }
    }

    /**
     * 发送无参数get请求
     * @param url 请求地址
     * @return 返回参数
     */
    public static String sendGet(String url) {
        return sendGet(url, null, null);
    }

    /**
     * 发送携带参数，没有请求头的get请求
     * @param url 请求地址
     * @param paramMap 请求参数
     * @return 返回参数
     */
    public static String sendGet(String url, Map<String, String> paramMap) {
        return sendGet(url, paramMap, null);
    }

}
