package top.cenze.rule.utils;

import cn.hutool.core.collection.CollectionUtil;
import lombok.extern.slf4j.Slf4j;
import org.drools.core.base.RuleNameStartsWithAgendaFilter;
import org.kie.api.KieBase;
import org.kie.api.KieServices;
import org.kie.api.builder.*;
import org.kie.api.runtime.KieContainer;
import org.kie.api.runtime.KieSession;
import top.cenze.rule.dto.RuleBaseData;

import java.util.List;
import java.util.Map;

/**
 * @desc: 规则管理工具
 * @author: chengze
 * @createByDate: 2023/8/8 17:30
 */
@Slf4j
public class KieUtil {
    // 规则文件虚拟加载目录路径
    private static final String KIE_PACKAGE_PATH = "src/main/resources/rules/";

    private static final KieServices kieServices = KieServices.Factory.get();
    // 创建kie文件系统
    private static final KieFileSystem kieFileSystem = kieServices.newKieFileSystem();
    private static KieContainer kieContainer;
    private static KieBase kieBase;

    public static KieFileSystem getKieFileSystem() {
        return kieFileSystem;
    }

    public static KieContainer getKieContainer() {
        return kieContainer;
    }

    public static void setKieContainer(KieContainer kieContainer) {
        KieUtil.kieContainer = kieContainer;
    }

    public static KieServices getKieServices() {
        return kieServices;
    }

    public static KieBase getKieBase() {
        return kieBase;
    }

    public static void setKieBase(KieBase kieBase) {
        KieUtil.kieBase = kieBase;
    }

    /**
     * 批量重载规则
     * @param lstRule
     */
    public static void reloadRules(List<RuleBaseData> lstRule) {
        if (CollectionUtil.isEmpty(lstRule)) {
            return;
        }

        for (RuleBaseData rule : lstRule) {
            log.info("rule name: {}", rule.getRuleName());
            String drlFile = KIE_PACKAGE_PATH + rule.getRuleName() + ".drl";
            kieFileSystem.delete(drlFile);
            kieFileSystem.write(drlFile, rule.getRuleContent());
        }

        KieBuilder kieBuilder = kieServices.newKieBuilder(kieFileSystem);
        kieBuilder.buildAll();
        Results results = kieBuilder.getResults();
        if (results.hasMessages(Message.Level.ERROR)) {
            throw new IllegalStateException("### load rules error ###");
        }

        kieContainer = kieServices.newKieContainer(kieServices.getRepository().getDefaultReleaseId());
        kieBase = kieContainer.getKieBase();
    }

//    /**
//     * 重载规则
//     * @param lstRule
//     */
//    public static void reloadRule(List<RuleDTO> lstRule) {
//        // 借助 kieHelper 添加规则
//        KieHelper kieHelper = new KieHelper();
//        // 从数据库中获取所有的规则
//        if (CollectionUtil.isNotEmpty(lstRule)) {
//            // 循环加载所有的规则
//            for (RuleDTO rule : lstRule) {
//                String content = rule.getRuleContent();
//                kieHelper.addContent(content, ResourceType.DRL);
//            }
//        }
//        // 校验规则是否异常
//        Results results = kieHelper.verify();
//        System.out.println("results: " + JSON.toJSONString(results));
//        if (results.hasMessages(Message.Level.ERROR)) {
//            System.out.println(results.getMessages());
//            throw new IllegalStateException("### errors ###");
//        }
//        // 重置容器
//        KieContainer kieContainer = kieHelper.getKieContainer();
//        KieUtil.setKieContainer(kieContainer);
//
//        // 重置规则库
//        KieBase kieBase = kieContainer.getKieBase();
//        KieUtil.setKieBase(kieBase);
//    }

    /**
     * 执行所有规则
     * @param obj
     */
    public static void execAll(Object obj) {
        KieSession kieSession = KieUtil.getKieBase().newKieSession();
        kieSession.insert(obj);
        kieSession.fireAllRules();
        kieSession.dispose();
    }

    /**
     * 执行所有规则
     * @param obj           待插入的Fact
     * @param mapGlobal     待传入的全局变量
     */
    public static void execAll(Object obj, Map<String, Object> mapGlobal) {
        KieSession kieSession = KieUtil.getKieBase().newKieSession();
        kieSession.insert(obj);
        if (CollectionUtil.isNotEmpty(mapGlobal)) {
            for (Map.Entry<String, Object> entry : mapGlobal.entrySet()) {
                kieSession.setGlobal(entry.getKey(), entry.getValue());
            }
        }
        kieSession.fireAllRules();
        kieSession.dispose();
    }

    /**
     * 执行指定名称的规则
     * drl规则文件中 rule 项对应的值
     * @param obj           待插入的Fact
     * @param ruleName      规则名（用于匹配包含此字符串的规则）
     */
    public static void execByRuleName(Object obj, String ruleName) {
        KieSession kieSession = KieUtil.getKieBase().newKieSession();
        kieSession.insert(obj);
        kieSession.fireAllRules(new RuleNameStartsWithAgendaFilter(ruleName));
        kieSession.dispose();
    }

    /**
     * 执行指定名称的规则
     * drl规则文件中 rule 项对应的值
     * @param obj           待插入的Fact
     * @param ruleName      规则名（用于匹配包含此字符串的规则）
     * @param mapGlobal     待传入的全局变量
     */
    public static void execByRuleName(Object obj, String ruleName, Map<String, Object> mapGlobal) {
        KieSession kieSession = KieUtil.getKieBase().newKieSession();
        kieSession.insert(obj);
        if (CollectionUtil.isNotEmpty(mapGlobal)) {
            for (Map.Entry<String, Object> entry : mapGlobal.entrySet()) {
                kieSession.setGlobal(entry.getKey(), entry.getValue());
            }
        }
        kieSession.fireAllRules(new RuleNameStartsWithAgendaFilter(ruleName));
        kieSession.dispose();
    }
}
