package top.cenze.rulepolicy.frame.config;

import org.kie.api.KieBase;
import org.kie.api.builder.KieBuilder;
import org.kie.api.builder.KieFileSystem;
import org.kie.api.builder.KieRepository;
import org.kie.api.runtime.KieContainer;
import org.kie.api.runtime.KieSession;
import org.kie.internal.io.ResourceFactory;
import org.kie.spring.KModuleBeanFactoryPostProcessor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternResolver;
import top.cenze.rulepolicy.frame.utils.KieUtil;

import java.io.IOException;

/**
 * @desc: 规则配置
 * @author: chengze
 * @createByDate: 2023/8/3 13:37
 */
@Configuration
public class DroolsConfig {

    //指定规则文件存放的目录
    public static final String RULES_PATH = "rules/";

    /**
     * 内存文件系统
     * @return
     * @throws IOException
     */
    @Bean
    @ConditionalOnMissingBean
    public KieFileSystem kieFileSystem() throws IOException {
        // 创建文件系统
        KieFileSystem kieFileSystem = KieUtil.getKieServices().newKieFileSystem();

        // 创建资源解释器
        ResourcePatternResolver resourcePatternResolver = new PathMatchingResourcePatternResolver();

        // 获取resource下RULES_PATH下所有资源文件
        Resource[] files = resourcePatternResolver.getResources("classpath*:" + RULES_PATH + "*.*");

        String path = null;
        for (Resource file : files) {
            path = RULES_PATH + file.getFilename();

            // 将资源文件载入至内存文件系统中
            kieFileSystem.write(ResourceFactory.newClassPathResource(path, "UTF-8"));
        }

        return kieFileSystem;
    }

    /**
     * 内存容器
     * @return
     * @throws IOException
     */
    @Bean
    @ConditionalOnMissingBean
    public KieContainer kieContainer() throws IOException {
        KieRepository kieRepository = KieUtil.getKieServices().getRepository();
        kieRepository.addKieModule(kieRepository::getDefaultReleaseId);
        KieBuilder kieBuilder = KieUtil.getKieServices().newKieBuilder(kieFileSystem());
        kieBuilder.buildAll();
        KieContainer kieContainer = KieUtil.getKieServices().newKieContainer(kieRepository.getDefaultReleaseId());

        KieUtil.setKieContainer(kieContainer);
        return kieContainer;
    }

    /**
     * 内存仓库
     * @return
     * @throws IOException
     */
    @Bean
    @ConditionalOnMissingBean
    public KieBase kieBase() throws IOException {
        KieBase kieBase = kieContainer().getKieBase();

        KieUtil.setKieBase(kieBase);
        return kieBase;
    }

    @Bean
    @ConditionalOnMissingBean
    public KieSession kieSession() throws IOException {
        KieSession kieSession = kieBase().newKieSession();

        return kieSession;
    }

    @Bean
    @ConditionalOnMissingBean
    public KModuleBeanFactoryPostProcessor kiePostProcessor() {
        return new KModuleBeanFactoryPostProcessor();
    }
}
