package top.cenze.rulepolicy.utils;

import cn.hutool.core.util.ObjectUtil;
import org.kie.api.runtime.KieSession;
import org.springframework.stereotype.Component;
import top.cenze.rulepolicy.entity.dto.FeeDTO;
import top.cenze.rulepolicy.enums.TaxWayEnum;
import top.cenze.rulepolicy.frame.utils.KieUtil;

import java.math.BigDecimal;
import java.math.RoundingMode;

/**
 * @desc: 费用规则工具
 * @author: chengze
 * @createByDate: 2023/8/9 15:43
 */
@Component
public class FeeRuleUtil {
    private static Integer PARAM_DEFAULT_SCALE = 6;     // 计算参数默认保留小数点位数
    private static Integer RESULT_DEFAULT_SCALE = 2;     // 计算结果默认保留小数点位数

    public static void calculate(FeeDTO dto) {
        // 格式化参数
        formatParam(dto);

        ruleFee(dto);
    }

    /**
     * 执行费用规则
     * @param dto
     */
    public static void ruleFee(FeeDTO dto) {
        KieSession kieSession = KieUtil.getKieBase().newKieSession();
        kieSession.insert(dto);
        kieSession.fireAllRules();
        kieSession.dispose();
    }

    /**
     * 校对计算参数，按指定小数点位数对参数进行格式化
     * @param dto
     */
    private static void formatParam(FeeDTO dto) {
        // 空值校验
        switch (dto.getFeeRule()) {
            case VOLUME:
                if (ObjectUtil.isNull(dto.getVolume())) {
                    throw new NullPointerException("volume is null");
                }

                if (ObjectUtil.isNull(dto.getVolumeScale())) {
                    dto.setVolumeScale(PARAM_DEFAULT_SCALE);
                }
                dto.setVolume(dto.getVolume().setScale(dto.getVolumeScale(), RoundingMode.HALF_UP));
                break;
            case WEIGHT:
                if (ObjectUtil.isNull(dto.getWeight())) {
                    throw new NullPointerException("weigth is null");
                }

                if (ObjectUtil.isNull(dto.getWeightScale())) {
                    dto.setWeightScale(PARAM_DEFAULT_SCALE);
                }
                dto.setWeight(dto.getWeight().setScale(dto.getWeightScale(), RoundingMode.HALF_UP));
                break;
            case QUANTITY:
                if (ObjectUtil.isNull(dto.getQuantity())) {
                    throw new NullPointerException("quantity is null");
                }

                if (ObjectUtil.isNull(dto.getQuantityScale())) {
                    dto.setQuantityScale(PARAM_DEFAULT_SCALE);
                }
                dto.setQuantity(dto.getQuantity().setScale(dto.getQuantityScale(), RoundingMode.HALF_UP));
                break;
        }
        if (ObjectUtil.isNotNull(dto.getTaxWay())) {
            if (ObjectUtil.isNull(dto.getTaxRate())) {
                throw new NullPointerException("taxRate is null");
            }

            if (ObjectUtil.isNull(dto.getTaxScale())) {
                dto.setTaxScale(PARAM_DEFAULT_SCALE);
            }
            dto.setTaxRate(dto.getTaxRate().setScale(dto.getTaxScale(), RoundingMode.HALF_UP));
        }
        // 如果计税方式为空，则默认为乘法，税率为0
        else {
            dto.setTaxWay(TaxWayEnum.MULTIPLY);
            dto.setTaxRate(BigDecimal.ZERO);
        }
        if (ObjectUtil.isNull(dto.getPrice())) {
            throw new NullPointerException("price is null");
        }

        // 小数位格式化
        if (ObjectUtil.isNull(dto.getScale())) {
            dto.setScale(RESULT_DEFAULT_SCALE);
        }

        if (ObjectUtil.isNull(dto.getPriceScale())) {
            dto.setPriceScale(PARAM_DEFAULT_SCALE);
        }
        dto.setPrice(dto.getPrice().setScale(dto.getPriceScale(), RoundingMode.HALF_UP));
    }
}
