package top.cenze.utils.http.request;

import cn.hutool.core.util.StrUtil;
import cn.hutool.crypto.digest.DigestUtil;
import cn.hutool.http.HttpRequest;
import cn.hutool.http.HttpUtil;
import cn.hutool.http.Method;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.parser.Feature;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.CollectionUtils;

import javax.net.ssl.*;
import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.*;

/**
 * https请求工具
 * author: cz
 */

@Slf4j
public class HttpsUtil {

    public static <T> T post(String url, Map<String, String> headers, Class<T> clazz) {
        return request(url, new HashMap<>(), headers, Method.POST, clazz);
    }

    public static <T> T postMap(String url, Map<String, Object> params, Map<String, String> headers, Class<T> clazz) {
        return request(url, params, headers, Method.POST, clazz);
    }

    public static <T> T postMap(String url, Map<String, Object> params, Class<T> clazz) {
        return request(url, params, null, Method.POST, clazz);
    }

    public static <T> T postObj(String url, Object obj, Class<T> clazz) {
        return request(url, JSON.toJSONString(obj), null, Method.POST, clazz);
    }

    public static <T> T postObj(String url, Object obj, Map<String, String> headers, Class<T> clazz) {
        return request(url, JSON.toJSONString(obj), headers, Method.POST, clazz);
    }

    public static <T> T postStr(String url, String str, Map<String, String> headers, Class<T> clazz) {
        return request(url, str, headers, Method.POST, clazz);
    }

    public static <T> T get(String url, Map<String, String> headers, Class<T> clazz) {
        return request(url, new HashMap<>(), headers, Method.GET, clazz);
    }

    public static <T> T getMap(String url, Map<String, Object> params, Map<String, String> headers, Class<T> clazz) {
        return request(url, params, headers, Method.GET, clazz);
    }

    public static <T> T getObj(String url, Object obj, Map<String, String> headers, Class<T> clazz) {
        return request(url, JSON.toJSONString(obj), headers, Method.GET, clazz);
    }

    public static <T> T getStr(String url, String str, Map<String, String> headers, Class<T> clazz) {
        return request(url, str, headers, Method.GET, clazz);
    }

    public static <T> T putObj(String url, Object obj, Map<String, String> headers, Class<T> clazz) {
        return request(url, JSON.toJSONString(obj), headers, Method.PUT, clazz);
    }

    public static <T> T del(String url, Map<String, String> headers, Class<T> clazz) {
        return request(url, "", headers, Method.DELETE, clazz);
    }

    public static <T> T delObj(String url, Object obj, Map<String, String> headers, Class<T> clazz) {
        return request(url, JSON.toJSONString(obj), headers, Method.DELETE, clazz);
    }

    public static String getCookie(String url, Map<String, Object> params) {
        HttpRequest httpRequest = HttpUtil.createGet(url);
        httpRequest.form(params);

        log.info("getCookie url: {}, req: {}", url, JSON.toJSONString(params));
        String result = httpRequest.execute()
                .getCookieStr();
        log.info("getCookie result: {}", result);

        return result;
    }

    public static <T> T request(String url, String str, Map<String, String> headers, Method method, Class<T> clazz) {
        if (null == method) {
            method = Method.POST;
        }

        HttpRequest httpRequest = HttpUtil.createRequest(method, url);
        httpRequest.setConnectionTimeout(60000000);

        if (!CollectionUtils.isEmpty(headers)) {
            for (Map.Entry<String, String> head : headers.entrySet()) {
                httpRequest.header(head.getKey(), head.getValue());
            }
        }

        if (!StringUtils.isBlank(str)) {
            httpRequest.body(str);
        }

        log.info("request url: {}, req: {}", url, str);
        String result = httpRequest.execute().body();
        log.info("request result: {}", result);

        if (!StringUtils.isBlank(result)) {
            if (null != clazz) {
                try {
                    return JSON.parseObject(result, clazz);
                } catch (Exception e) {
                    log.error("request err: {}, result: {}", e.getMessage(), result);
                }
            } else {
                return (T) result;
            }
        }

        return null;
    }

    public static <T> T request(String url, Map<String, Object> params, Map<String, String> headers, Method method, Class<T> clazz) {
        if (null == method) {
            method = Method.POST;
        }

        HttpRequest httpRequest = HttpUtil.createRequest(method, url);

        if (!CollectionUtils.isEmpty(headers)) {
            for (Map.Entry<String, String> head : headers.entrySet()) {
                httpRequest.header(head.getKey(), head.getValue());
            }
        }

        if (!CollectionUtils.isEmpty(params)) {
            httpRequest.form(params);
        }

        log.info("request url: {}, req: {}", url, JSON.toJSONString(params));
        String result = httpRequest.execute().body();
        log.info("request result: {}", result);

        if (!StringUtils.isBlank(result)) {
            if (null != clazz) {
                try {
                    return JSON.parseObject(result, clazz);
                } catch (Exception e) {
                    log.error("request err: {}, result: {}", e.getMessage(), result);
                }
            } else {
                return (T) result;
            }
        }

        return null;
    }


    /**
     * 发送HTTPS请求
     * 访问https资源时,忽略证书信任问题
     * @param toURL 请求地址
     * @param data 请求参数
     * @param dateType 参数类型，“JSON”,other
     * @return
     * @throws Exception
     */
    public static String httpsPost(String toURL, String data, String dateType)
            throws Exception {
        StringBuffer bs = new StringBuffer();

        //  直接通过主机认证
        HostnameVerifier hv = new HostnameVerifier() {
            @Override
            public boolean verify(String arg0, SSLSession arg1) {
                return true;
            }
        };
        //  配置认证管理器
        javax.net.ssl.TrustManager[] trustAllCerts = {new TrustAllTrustManager()};
        SSLContext sc = SSLContext.getInstance("SSL");
        SSLSessionContext sslsc = sc.getServerSessionContext();
        sslsc.setSessionTimeout(0);
        sc.init(null, trustAllCerts, null);
        HttpsURLConnection.setDefaultSSLSocketFactory(sc.getSocketFactory());
        //  激活主机认证
        HttpsURLConnection.setDefaultHostnameVerifier(hv);
        URL url = new URL(toURL);
        HttpURLConnection connection = (HttpURLConnection)url.openConnection();
        connection.setConnectTimeout(30000);
        connection.setReadTimeout(30000);
        connection.setDoOutput(true);
        if("JSON".equalsIgnoreCase(dateType)){
            connection.setRequestProperty("Content-Type","application/json;chert=UTF-8");
        }else{
            connection.setRequestProperty("Content-Type","application/x-www-form-urlencoded");
        }
        OutputStreamWriter out = new OutputStreamWriter(
                connection.getOutputStream(), "UTF-8");
        out.write(data);
        out.flush();
        out.close();
        connection.connect();
        int code = ((HttpURLConnection) connection).getResponseCode();
        InputStream is =  null;
        if (code == 200) {
            is = connection.getInputStream(); // 得到网络返回的正确输入流
        } else {
            is = ((HttpURLConnection) connection).getErrorStream(); // 得到网络返回的错误输入流
        }
        BufferedReader buffer = new BufferedReader(new InputStreamReader(is,"UTF-8"));

        String l = null;
        while ((l = buffer.readLine()) != null) {
            bs.append(l);
        }
        //return bs.toString();
        return String.valueOf(bs);
    }

    /**
     * 对象转有map
     *
     * @param obj
     * @return
     */
    public static Map<String, Object> objToMap(Object obj) {
        return JSON.parseObject(JSON.toJSONString(obj), HashMap.class, Feature.OrderedField);
    }

    public static StringBuilder mapToSortStr(Map<String, Object> params) {
        // 移除值为空的
        params.entrySet().removeIf(
                entry -> Objects.isNull(entry.getValue()) ||
                "".equals(entry.getValue()));

        List<Map.Entry<String, Object>> infoIds = new ArrayList<>(params.entrySet());
        // 对所有传入参数按照字段名的ASCII码从小到大排序（字典序）
        infoIds.sort((o1, o2) -> o1.getKey().compareToIgnoreCase(o2.getKey()));
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<String, Object> infoId : infoIds) {
            sb.append(infoId.getKey());
            sb.append("=");
            sb.append(infoId.getValue());
            sb.append("&");
        }

        // 删除末尾的 "&"
        sb.deleteCharAt(sb.length() - 1);

        return sb;
    }

    /**
     * map按Key顺序排序
     * @param params
     * @return
     */
    public static LinkedHashMap<String, Object> mapToSortMap(Map<String, Object> params) {
        LinkedHashMap<String, Object> map = new LinkedHashMap<>();

        // 移除值为空的
        params.entrySet().removeIf(
                entry -> Objects.isNull(entry.getValue()) ||
                        "".equals(entry.getValue()));

        List<Map.Entry<String, Object>> infoIds = new ArrayList<>(params.entrySet());

        // 对所有传入参数按照字段名的ASCII码从小到大排序（字典序）
        infoIds.sort((o1, o2) -> o1.getKey().compareToIgnoreCase(o2.getKey()));
        for (Map.Entry<String, Object> infoId : infoIds) {
            map.put(infoId.getKey(), infoId.getValue());
        }

        return map;
    }

    /**
     * obj 转 mac buff
     * @param obj
     * @return
     */
    public static String objToMacBuff(Object obj) {
        StringBuilder sb = new StringBuilder();

        Map<String, Object> mapParams = objToMap(obj);

        LinkedHashMap<String, Object> map = mapToSortMap(mapParams);

        for (Map.Entry<String, Object> e : mapParams.entrySet()) {
            sb.append(" ").append(e.getValue());
        }

        return StrUtil.trim(sb.toString());
    }

    /**
     * 获取排序参数签名
     * @param key
     * @return
     * @throws UnsupportedEncodingException
     */
    public static String getSign(StringBuilder sb, String key) throws UnsupportedEncodingException {
        // 最后拼接上key得到signTemp字符串
        String signTemp = sb.append(key).toString();

        // 对signTemp进行SHA-256运算，再将得到的字符串所有字符转换为大写，得到sign值signValue
        String signValue = DigestUtil.sha256Hex(signTemp).toUpperCase();

        return signValue;
    }

    /**
     * 获取排序参数签名
     * @param key
     * @return
     * @throws UnsupportedEncodingException
     */
    public static String getSign(Object obj, String key) throws UnsupportedEncodingException {
        Map<String, Object> params = objToMap(obj);

        StringBuilder sb = mapToSortStr(params);

        String sign = getSign(sb, key);

        return sign;
    }

    /**
     * 获取排序参数签名（含签名参数）
     * @param obj
     * @param key
     * @return
     * @throws UnsupportedEncodingException
     */
    public static String getParamsAndSign(Object obj, String key) throws UnsupportedEncodingException {
        Map<String, Object> params = objToMap(obj);

        StringBuilder sb = mapToSortStr(params);

        String sign = getSign(sb, key);

        sb.append("&sign=").append(sign);

        return sb.toString();
    }
}
