package top.cenze.utils.http.request;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.convert.Convert;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.netflix.zuul.context.RequestContext;
import com.netflix.zuul.http.ServletInputStreamWrapper;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.StreamUtils;

import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Slf4j
public class ZuulUtil {

    // 多次反向代理后会有多个ip值 的分割符
    private final static String IP_UTILS_FLAG = ",";
    // 未知IP
    private final static String UNKNOWN = "unknown";
    // 本地 IP
    private final static String LOCALHOST_IP = "0:0:0:0:0:0:0:1";
    private final static String LOCALHOST_IP1 = "127.0.0.1";

    /**
     * 获取请求TOKEN
     * @param ctx
     * @return
     */
    public static String getToken(RequestContext ctx) {
        // 请求
        HttpServletRequest request = ctx.getRequest();

        String token = request.getHeader("TOKEN");
        if (StringUtils.isBlank(token)) {
            token = getParameter(ctx, "TOKEN");
        }

        return token;
    }

    /**
     * 获取请求参数
     * @param ctx
     * @param name
     * @return
     */
    @SneakyThrows
    public static String getParameter(RequestContext ctx, String name) {
        // 请求
        HttpServletRequest request = ctx.getRequest();
        // 请求方法
        String method = request.getMethod();

        // get方法，和post、put方法处理方式不同
        if ("GET".equals(method)) {
            // 一定要在获取参数前设置一下编码，防止获取的中文乱码
            request.setCharacterEncoding("UTF-8");
            // 关键步骤，一定要get一下,下面才能取到值requestQueryParams
            request.getParameterMap();
            Map<String, List<String>> requestQueryParams = ctx.getRequestQueryParams();
            if (requestQueryParams == null) {
                requestQueryParams = new HashMap<>();
            }
            log.info("getParameter requestQueryParams: {}", JSON.toJSONString(requestQueryParams));

            // 获取参数的第一个值
            List<String> lstParam = requestQueryParams.get(name);
            if (CollectionUtil.isNotEmpty(lstParam)) {
                return lstParam.get(0);
            }
        }
        // post和put需重写HttpServletRequestWrapper
        else if ("POST".equals(method) || "PUT".equals(method)) {
            // 获取请求的输入流
            InputStream in = request.getInputStream();
            String body = StreamUtils.copyToString(in, Charset.forName("UTF-8"));
            // 如果body为空初始化为空json
            if (StrUtil.isEmpty(body)) {
                body = "{}";
            }
            log.info("getParameter body: {}", body);
            // 转化成json
            JSONObject json = JSONObject.parseObject(body);

            if (ObjectUtil.isNotNull(json) && ObjectUtil.isNotNull(json.get(name))) {
                return Convert.toStr(json.get(name));
            }
        }

        return null;
    }

    /**
     * 添加修改请求参数
     * @param ctx
     * @param name
     * @param value
     */
    @SneakyThrows
    public static void setParameter(RequestContext ctx, String name, Object value) {
        Map<String, Object> mapParams = new HashMap<>();
        mapParams.put(name, value);

        setParameters(ctx, mapParams);
    }

    /**
     * 添加修改请求参数集合
     * @param ctx
     * @param mapParams
     */
    @SneakyThrows
    public static void setParameters(RequestContext ctx, Map<String, Object> mapParams) {
        // 请求
        HttpServletRequest request = ctx.getRequest();
        // 请求方法
        String method = request.getMethod();

        // get方法，和post、put方法处理方式不同
        if ("GET".equals(method)) {
            // 一定要在获取参数前设置一下编码，防止获取的中文乱码
            request.setCharacterEncoding("UTF-8");
            // 关键步骤，一定要get一下,下面才能取到值requestQueryParams
            request.getParameterMap();
            Map<String, List<String>> requestQueryParams = ctx.getRequestQueryParams();
            if (requestQueryParams == null) {
                requestQueryParams = new HashMap<>();
            }
            log.info("setParameter requestQueryParams: {}", JSON.toJSONString(requestQueryParams));

            // 添加/修改参数
            if (CollectionUtil.isNotEmpty(mapParams)) {
                for (Map.Entry<String, Object> entry : mapParams.entrySet()) {
                    requestQueryParams.put(entry.getKey(), Arrays.asList(Convert.toStr(entry.getValue())));
                }

                log.info("setParameter newQueryParams: {}", JSON.toJSONString(requestQueryParams));
                ctx.setRequestQueryParams(requestQueryParams);
            }
        }
        // post和put需重写HttpServletRequestWrapper
        else if ("POST".equals(method) || "PUT".equals(method)) {
            // 获取请求的输入流
            InputStream in = request.getInputStream();
            String body = StreamUtils.copyToString(in, Charset.forName("UTF-8"));
            // 如果body为空初始化为空json
            if (StrUtil.isEmpty(body)) {
                body = "{}";
            }
            log.info("setParameter body: {}", body);
            // 转化成json
            JSONObject json = JSONObject.parseObject(body);

            // 添加/修改参数
            if (CollectionUtil.isNotEmpty(mapParams)) {
                for (Map.Entry<String, Object> entry : mapParams.entrySet()) {
                    json.put(entry.getKey(), entry.getValue());
                }

                String newBody = json.toString();
                log.info("setParameter newBody: {}", newBody);
                final byte[] reqBodyBytes = newBody.getBytes();

                // 重写上下文的HttpServletRequestWrapper
                ctx.setRequest(new HttpServletRequestWrapper(request) {
                    @Override
                    public ServletInputStream getInputStream() throws IOException {
                        return new ServletInputStreamWrapper(reqBodyBytes);
                    }

                    @Override
                    public int getContentLength() {
                        return reqBodyBytes.length;
                    }

                    @Override
                    public long getContentLengthLong() {
                        return reqBodyBytes.length;
                    }
                });
            }
        }
    }

    /**
     * 获取访问者真实ip
     * @param ctx
     * @return
     */
    public static String getRemoteIP(RequestContext ctx) {
        // 获取请求
        HttpServletRequest request = ctx.getRequest();

        return getRemoteIP(request);
    }

    /**
     * 获取访问者真实ip
     * @param request
     * @return
     */
    public static String getRemoteIP(HttpServletRequest request) {
        String ip = null;
        try {
            // 以下两个获取在k8s中，将真实的客户端IP，放到了x-Original-Forwarded-For。而将WAF的回源地址放到了 x-Forwarded-For了。
            ip = request.getHeader("X-Original-Forwarded-For");
            if (StrUtil.isEmpty(ip) || UNKNOWN.equalsIgnoreCase(ip)) {
                ip = request.getHeader("X-Forwarded-For");
            }
            // 获取nginx等代理的ip
            if (StrUtil.isEmpty(ip) || UNKNOWN.equalsIgnoreCase(ip)) {
                ip = request.getHeader("x-forwarded-for");
            }
            if (StrUtil.isEmpty(ip) || UNKNOWN.equalsIgnoreCase(ip)) {
                ip = request.getHeader("Proxy-Client-IP");
            }
            if (StrUtil.isEmpty(ip) || ip.length() == 0 || UNKNOWN.equalsIgnoreCase(ip)) {
                ip = request.getHeader("WL-Proxy-Client-IP");
            }
            if (StrUtil.isEmpty(ip) || UNKNOWN.equalsIgnoreCase(ip)) {
                ip = request.getHeader("HTTP_CLIENT_IP");
            }
            if (StrUtil.isEmpty(ip) || UNKNOWN.equalsIgnoreCase(ip)) {
                ip = request.getHeader("HTTP_X_FORWARDED_FOR");
            }
            // 兼容k8s集群获取ip
            if (StrUtil.isEmpty(ip) || UNKNOWN.equalsIgnoreCase(ip)) {
                ip = request.getRemoteAddr();
                if (LOCALHOST_IP1.equalsIgnoreCase(ip) || LOCALHOST_IP.equalsIgnoreCase(ip)) {
                    // 根据网卡取本机配置的IP
                    InetAddress iNet = null;
                    try {
                        iNet = InetAddress.getLocalHost();
                    } catch (UnknownHostException e) {
                        log.error("getClientIp error: ", e);
                    }
                    ip = iNet.getHostAddress();
                }
            }
        } catch (Exception e) {
            log.error("IPUtils ERROR ", e);
        }

        // 使用代理，则获取第一个IP地址
        if (StrUtil.isNotEmpty(ip) && ip.indexOf(IP_UTILS_FLAG) > 0) {
            ip = ip.substring(0, ip.indexOf(IP_UTILS_FLAG));
        }

        return ip;
    }

    /**
     * 获取访问域名
     * @param ctx
     * @return
     */
    public static String getServerName(RequestContext ctx) {
        // 获取请求
        HttpServletRequest request = ctx.getRequest();

        return getServerName(request);
    }

    /**
     * 获取访问域名
     * @param request
     * @return
     */
    public static String getServerName(HttpServletRequest request) {
        return request.getServerName();
    }
}
