package top.cenze.utils.file;

import cn.hutool.core.io.FileUtil;
import cn.hutool.core.io.resource.ResourceUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import top.cenze.utils.enums.ExcelTypeEnum;
import top.cenze.utils.enums.ReportFileTypeEnum;

import java.io.*;
import java.util.UUID;

/**
 * @desc: 通用文件工具
 * @author: chengze
 * @createByDate: 2023/10/12 9:06
 */
@Slf4j
public class CZFileUtil {
    private static final String TEMPLATE_DIR = "template/";
    private static final String REPORT_DIR = "report/";

    /**
     * 获取当前操作系统名称. return 操作系统名称 例如:windows,Linux,Unix等.
     */
    public static String getOSName() {
        return System.getProperty("os.name").toLowerCase();
    }

    /**
     * 加载文件
     * @param filePath
     * @return
     * @throws FileNotFoundException
     */
    @SneakyThrows
    public static InputStream readFile(String filePath) {
        log.info("loadFile filePath: {}", filePath);
        boolean exist = FileUtil.exist(filePath);
        log.info("loadFile exist: {}", exist);
        if (!exist) {
            throw new Exception("file is not exist!");
        }

        return new FileInputStream(filePath);
    }

    /**
     * 获取项目resource目录结对路径
     * @return
     */
    public static String getResourcePath() {
        String path = ResourceUtil.getResource("").getPath();
        log.info("getResourcePath path: {}", path);
//        String path = CZFileUtil.class.getResource("/").getPath();
//
//        String osName = getOSName().toLowerCase();
//        if (StrUtil.isNotEmpty(path) && path.length() > 0) {
//            if ("windows".equals(osName)) {
//                if ("/".equals(path.substring(0, 1))) {
//                    path = path.substring(1);
//                }
//            } else {
//                int i = path.indexOf(":");
//                if (i >= 0) {
//                    path = path.substring(i + 1);
//                }
//            }
//        }
//        path = StrUtil.trim(path);
//
//        log.info("getResourcePath path: {}", path);

        return path;
    }

    /**
     * 在项目resource创建文件
     * 默认目录：template/
     * @param directory
     * @param fileName
     */
    public static File mkFileToResource(String directory, String fileName){
        if (StrUtil.isEmpty(directory)) {
            directory = TEMPLATE_DIR;
        }
        log.info("mkFileToResource dir: {}, fileName: {}", directory, fileName);
        File file = null;

        directory = getResourcePath() + directory;
        if (!FileUtil.exist(directory)){
            FileUtil.mkdir(directory);
        }

        String filePath = directory + fileName;
        if(!FileUtil.exist(filePath)){
            file = FileUtil.touch(filePath);
        } else {
            file = FileUtil.file(filePath);
        }

        return file;
    }

    /**
     * 向项目resource写入文件
     * @param directory
     * @param fileName
     * @param content
     */
    public static void writeFileToResource(String directory, String fileName, String content){
        File file = mkFileToResource(directory, fileName);

        writeFileToResource(file, content);
    }

    /**
     * 向项目resource写入文件
     * @param file
     * @param content
     */
    public static void writeFileToResource(File file, String content){

        try {
            FileWriter fw = new FileWriter(file);

            fw.write(content);
            fw.close();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 向项目resource写入文件
     * @param directory
     * @param fileName
     * @param content
     */
    public static void writeFileToResource(String directory, String fileName, byte[] content){
        File file = mkFileToResource(directory, fileName);

        writeFileToResource(file, content);
    }

    /**
     * 向项目resource写入文件
     * @param file
     * @param content
     */
    public static void writeFileToResource(File file, byte[] content){

        try {
            // 参数1：文件路径
            // 参数2：是否追加写入（true为追加，false为覆盖）
            FileOutputStream fos = new FileOutputStream(file, false);

            fos.write(content);
            fos.close();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 创建一个临时Excel文件
     * @return
     */
    public static File createExcelTemplateFile() {
        String templateFileName = UUID.randomUUID().toString().replaceAll("-", "") + ExcelTypeEnum.XLSX.getExtname();
        File tmpFile = CZFileUtil.mkFileToResource(TEMPLATE_DIR, templateFileName);
        if (!FileUtil.exist(tmpFile)) {
            throw new NullPointerException("创建模板文件失败");
        }

        return tmpFile;
    }

    /**
     * 创建一个临时Xml文件
     * @return
     */
    public static File createXmlTemplateFile() {
        String templateFileName = UUID.randomUUID().toString().replaceAll("-", "") + ".xml";
        File tmpFile = CZFileUtil.mkFileToResource(TEMPLATE_DIR, templateFileName);
        if (!FileUtil.exist(tmpFile)) {
            throw new NullPointerException("创建模板文件失败");
        }

        return tmpFile;
    }

    /**
     * 创建一个临时报表文件
     * @param fileType
     * @return
     */
    public static  File createReportFIle(ReportFileTypeEnum fileType) {
        String fileName = UUID.randomUUID().toString().replaceAll("-", "") + fileType.getExtname();
        File tmpFile = CZFileUtil.mkFileToResource(REPORT_DIR, fileName);
        if (!FileUtil.exist(tmpFile)) {
            throw new NullPointerException("创建报表文件失败");
        }

        return tmpFile;
    }

    /**
     * 文件转byte[]
     * @param file
     * @return
     */
    @SneakyThrows
    public static byte[] file2Bytes(File file) {
        if (ObjectUtil.isNull(file)) {
            return null;
        }

        FileInputStream fis = new FileInputStream(file);
        BufferedInputStream bis = new BufferedInputStream(fis);
        byte[] buf = new byte[bis.available()];
        bis.read(buf);
        bis.close();

        return buf;
    }
}
