package top.cenze.utils.aspect;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.reflect.CodeSignature;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.core.annotation.AnnotationFilter;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

/**
 * AOP工具
 */
@Slf4j
public class AspectUtil {
    /**
     * 获取所有请求参数
     * 支持基础类型与对象及自定义对象混合情况，如果是对象或自定义对象，提取其成员变量
     * @param point
     * @return
     */
    public static Map<String, Object> getRequestAllParams(JoinPoint point) {
        Map<String, Object> params = getRequestParams(point);
        log.info("getRequestAllParams params: {}", JSON.toJSONString(params));

        Map<String, Object> allParams = new HashMap<>();
        if (CollectionUtil.isNotEmpty(allParams)) {
            for (Map.Entry<String, Object> entry : allParams.entrySet()) {
                String name = entry.getKey();
                log.info("getRequestAllParams name: {}", name);
                Object obj = entry.getValue();
                log.info("getRequestAllParams obj: {}", JSON.toJSONString(obj));
                // java基础类型
                if (isJavaLang(obj)) {
                    params.put(name, obj);
                }
                // 对象或自定义对象
                else {
                    Map<String, Object> paramsByObj = getParamsByObj(obj);
                    log.info("getRequestAllParams paramsByObj: {}", JSON.toJSONString(paramsByObj));
                    if (CollectionUtil.isNotEmpty(paramsByObj)) {
                        allParams.putAll(paramsByObj);
                    }
                }
            }
        }

        return allParams;
    }

    /**
     * 获取所有请求参数
     * 支持基础类型与对象及自定义对象混合情况，不提取对象或自定义对象内的成员变量
     * @param point
     * @return
     */
    public static Map<String, Object> getRequestParams(JoinPoint point) {
        Map<String, Object> params = new HashMap<>();

        MethodSignature signature = (MethodSignature) point.getSignature();
        log.info("getRequestParams signature: {}", JSON.toJSONString(signature));
//        Method method = signature.getMethod();

        // 请求的方法所有参数值
        Object[] args = point.getArgs();
        log.info("getRequestParams args: {}", JSON.toJSONString(args));
        // 请求的方法所有参数名称
        String[] paramNames = ((CodeSignature)signature).getParameterNames();
        // 请求的方法参数名称
//        LocalVariableTableParameterNameDiscoverer u = new LocalVariableTableParameterNameDiscoverer();
//        String[] paramNames = u.getParameterNames(method);
        log.info("getRequestParams paramNames: {}", JSON.toJSONString(paramNames));

        if (ObjectUtil.isNotEmpty(args) && ObjectUtil.isNotEmpty(paramNames)) {
            for (int i = 0; i < args.length; i++) {
                Object obj = args[i];
                // 请求参数
                if (obj instanceof HttpServletRequest) {
                    String name = paramNames[i];
                    params.put(name, obj);
                }
            }
        }

        return params;
    }

    public static Object getParamByName(JoinPoint point, String name) {
        MethodSignature signature = (MethodSignature) point.getSignature();
        log.info("getParamByName signature: {}", JSON.toJSONString(signature));

        // 请求的方法所有参数值
        Object[] args = point.getArgs();
        log.info("getParamByName args: {}", JSON.toJSONString(args));
        if (ObjectUtil.isNotEmpty(args)) {
            for (Object obj : args) {
                JSONObject jsonObject = JSON.parseObject(JSON.toJSONString(obj));
                log.info("getParamByName jsonObject: {}", JSON.toJSONString(jsonObject));
                if (jsonObject.containsKey(name)) {
                    return jsonObject.get(name);
                }
            }
        }

        return null;
    }

    /**
     * 获取所有响应参数
     * 支持基础类型与对象及自定义对象混合情况，如果是对象或自定义对象，提取其成员变量
     * @param point
     * @return
     */
    public static Map<String, Object> getResponseAllParams(JoinPoint point) {
        Map<String, Object> params = getResponseParams(point);

        Map<String, Object> allParams = new HashMap<>();
        if (CollectionUtil.isNotEmpty(allParams)) {
            for (Map.Entry<String, Object> entry : allParams.entrySet()) {
                String name = entry.getKey();
                Object obj = entry.getValue();
                // java基础类型
                if (isJavaLang(obj)) {
                    params.put(name, obj);
                }
                // 对象或自定义对象
                else {
                    Map<String, Object> paramsByObj = getParamsByObj(obj);
                    if (CollectionUtil.isNotEmpty(paramsByObj)) {
                        allParams.putAll(paramsByObj);
                    }
                }
            }
        }

        return allParams;
    }

    /**
     * 获取所有响应参数
     * 支持基础类型与对象及自定义对象混合情况，不提取对象或自定义对象内的成员变量
     * @param point
     * @return
     */
    public static Map<String, Object> getResponseParams(JoinPoint point) {
        Map<String, Object> params = new HashMap<>();

        MethodSignature signature = (MethodSignature) point.getSignature();

        // 请求的方法所有参数值
        Object[] args = point.getArgs();
        // 请求的方法所有参数名称
        String[] paramNames = ((CodeSignature)signature).getParameterNames();

        if (ObjectUtil.isNotEmpty(args) && ObjectUtil.isNotEmpty(paramNames)) {
            for (int i = 0; i < args.length; i++) {
                Object obj = args[i];
                // 请求参数
                if (obj instanceof HttpServletResponse) {
                    String name = paramNames[i];
                    params.put(name, obj);
                }
            }
        }

        return params;
    }

    /**
     * 获取对象内成员变量Map
     * @param obj
     * @return
     */
    public static Map<String, Object> getParamsByObj(Object obj) {
        Map<String, Object> params = new HashMap<>();

        Field[] fields = obj.getClass().getDeclaredFields();
        if (ObjectUtil.isNotEmpty(fields)) {
            for (Field field : fields) {
                field.setAccessible(true);
                String key = field.getName();
                try {
                    Object val = field.get(obj);

                    if (ObjectUtil.isNotEmpty(val)) {
                        params.put(key, val);
                    }
                } catch (IllegalAccessException e) {
                    e.printStackTrace();
                }
            }
        }

        return params;
    }

    @SneakyThrows
    public static void setParamByName(JoinPoint point, String name, Object val) {
        MethodSignature methodSignature = (MethodSignature) point.getSignature();
        final String[] names = methodSignature.getParameterNames(); // 获取参数的名字
        log.info("setParamByName names: {}", JSON.toJSONString(names));
        final Object[] args = point.getArgs(); // 获取参数的值
        log.info("setParamByName args: {}", JSON.toJSONString(args));

        if (ObjectUtil.isNull(names) || ObjectUtil.isNull(args) ||
                0 == names.length || 0 == args.length) {
            return;
        }

        // 直接查看参数名匹配（参数为基础类型，直接设置值）
        int idx = ArrayUtil.indexOfIgnoreCase(names, name);
        log.info("setParamByName idx: {}", idx);
        if (idx >= 0) {
            // 如果是基础类型，直接设置值
            if (isJavaLang(args[idx])) {
                args[idx] = val;
            }
        }
        // 反射方式查找，并修改值
        else {
            for (Object obj : args) {
                Method setPlatformIdMethod = obj.getClass().getMethod("setPlatformId", Long.class);
                log.info("setParamByName args2: {}", JSON.toJSONString(args));
                if (ObjectUtil.isNull(setPlatformIdMethod)) {
                    continue;
                }

                setPlatformIdMethod.invoke(obj, val);
                log.info("setParamByName args3: {}", JSON.toJSONString(args));
            }
        }
    }

    /**
     * 判断是否为JAVA基础类型
     * @param obj
     * @return
     */
    public static boolean isJavaLang(Object obj) {
        return AnnotationFilter.PLAIN.matches(obj.getClass());
//        return obj.getClass().getName().startsWith("java.lang");
    }
}
