package top.cenze.utils.file;

import lombok.extern.slf4j.Slf4j;
import net.coobird.thumbnailator.Thumbnails;
import org.apache.commons.lang.StringUtils;
import sun.misc.BASE64Encoder;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;

@Slf4j
public class ImageUtil {
    private static final Integer ZERO = 0;
    private static final Integer ONE_ZERO_TWO_FOUR = 1024;
    private static final Integer NINE_ZERO_ZERO = 900;
    private static final Integer THREE_TWO_SEVEN_FIVE = 3275;
    private static final Integer TWO_ZERO_FOUR_SEVEN = 2047;
    private static final Double ZERO_EIGHT_FIVE = 0.85;
    private static final Double ZERO_SIX = 0.6;
    private static final Double ZERO_FOUR_FOUR = 0.44;
    private static final Double ZERO_FOUR = 0.4;

    /**
     * 根据指定大小压缩图片
     *
     * @param imageBytes  源图片字节数组
     * @param desFileSize 指定图片大小，单位kb
     * @return 压缩质量后的图片字节数组
     */
    public static byte[] compressPicForScale(byte[] imageBytes, long desFileSize) {
        if (imageBytes == null || imageBytes.length <= ZERO || imageBytes.length < desFileSize * ONE_ZERO_TWO_FOUR) {
            return imageBytes;
        }

        long srcSize = imageBytes.length;
        double accuracy = getAccuracy(srcSize / ONE_ZERO_TWO_FOUR);

        try {
            while (imageBytes.length > desFileSize * ONE_ZERO_TWO_FOUR) {
                ByteArrayInputStream inputStream = new ByteArrayInputStream(imageBytes);
                ByteArrayOutputStream outputStream = new ByteArrayOutputStream(imageBytes.length);

                Thumbnails.of(inputStream)
                        .scale(accuracy)
                        .outputQuality(accuracy)
                        .toOutputStream(outputStream);

                imageBytes = outputStream.toByteArray();
            }

            log.info("图片原大小={}kb | 压缩后大小={}kb",
                    srcSize / ONE_ZERO_TWO_FOUR, imageBytes.length / ONE_ZERO_TWO_FOUR);
        } catch (Exception e) {
            log.error("【图片压缩】msg=图片压缩失败!", e);
        }

        return imageBytes;
    }

    /**
     * 自动调节精度(经验数值)
     *
     * @param size 源图片大小
     * @return 图片压缩质量比
     */
    private static double getAccuracy(long size) {
        double accuracy;
        if (size < NINE_ZERO_ZERO) {
            accuracy = ZERO_EIGHT_FIVE;
        } else if (size < TWO_ZERO_FOUR_SEVEN) {
            accuracy = ZERO_SIX;
        } else if (size < THREE_TWO_SEVEN_FIVE) {
            accuracy = ZERO_FOUR_FOUR;
        } else {
            accuracy = ZERO_FOUR;
        }
        return accuracy;
    }

    /**
     * 判断是否为网络图片
     * @param image
     * @return
     */
    public static boolean isNetImage(String image) {
        if (StringUtils.isBlank(image)) {
            return false;
        }

        String str = image.substring(0, 10).toLowerCase();
        if (str.contains("http:") || str.contains("https:")) {
            return true;
        }

        return false;
    }

    /**
     * 网络图片转byte[]
     * @param imageUrl
     * @return
     */
    public static byte[] netImageToByte(String imageUrl) {
        if (org.apache.commons.lang3.StringUtils.isBlank(imageUrl)) {
            return null;
        }

        try {
            URL url = new URL(imageUrl);
            byte[] b = new byte[10240];

            HttpURLConnection conn = (HttpURLConnection) url.openConnection();
            conn.setRequestMethod("GET");
            conn.setConnectTimeout(5000);

            InputStream is = conn.getInputStream();
            ByteArrayOutputStream os = new ByteArrayOutputStream();

            int len = -1;
            while ((len = is.read(b)) != -1) {
                os.write(b, 0, len);
            }

            is.close();

            return os.toByteArray();
        } catch (Exception e) {
            log.error(e.getMessage());
        }

        return null;
    }

    /**
     * 网络图片转base64（网络图片地址、图片类型png、不需要base64头信息）
     * @param imageUrl
     * @return
     */
    public static String netImage2Base64(String imageUrl) {
        return netImage2Base64(imageUrl, false);
    }

    /**
     * 网络图片转base64（网络图片地址、图片类型png、是否需要base64头信息、不限制图片大小）
     * @param imageUrl
     * @param head      是否需要base64头
     * @return
     */
    public static String netImage2Base64(String imageUrl, boolean head) {
        return netImage2Base64(imageUrl, head, 0L);
    }

    /**
     * 网络图片转base64（网络图片地址、图片类型png、是否需要base64头信息、压缩图片限制压缩后的图片大小）
     * @param imageUrl
     * @param head          是否需要base64头
     * @param desFileSize   压缩图片，限制压缩后的图片大小
     * @return
     */
    public static String netImage2Base64(String imageUrl, boolean head, long desFileSize) {
        if (org.apache.commons.lang3.StringUtils.isBlank(imageUrl)) {
            return null;
        }

        String[] split = imageUrl.split(".");

        String type = "png";
        if (null != split && split.length > 0) {
            type = split[split.length - 1];
        }

        byte[] imageBytes = netImageToByte(imageUrl);

        if (desFileSize > 0) {
            imageBytes = compressPicForScale(imageBytes, desFileSize);
        }

        return netImage2Base64(imageBytes, type, head);
    }

    private static String netImage2Base64(byte[] imageBytes, String type, boolean head) {
        if (imageBytes == null || imageBytes.length <= ZERO) {
            return null;
        }

        try {
            BASE64Encoder encoder = new BASE64Encoder();
            if (head) {
                return "data:image/" + type + ";base64," + encoder.encode(imageBytes);
            } else {
                return encoder.encode(imageBytes);
            }
        } catch (Exception e) {
            log.error(e.getMessage());
        }

        return null;
    }

    /**
     * 网络图片转base64（网络图片地址、图片类型默认、不需要base64头信息）
     * @param imageUrl
     * @return
     */
    public static String NetImage2Base64(String imageUrl) {
        return NetImage2Base64(imageUrl, null, false);
    }

    /**
     * 网络图片转base64（网络图片地址、图片类型png、需要base64头信息）
     * @param imageUrl
     * @return
     */
    public static String NetImage2FullBase64(String imageUrl) {
        return NetImage2Base64(imageUrl, "png", true);
    }

    /**
     * 网络图片转base64（网络图片地址、图片类型png/jpg、需要base64头信息）
     * @param imageUrl
     * @param type
     * @return
     */
    public static String NetImage2Base64(String imageUrl, String type) {
        return NetImage2Base64(imageUrl, type, true);
    }

    /**
     * 网络图片转base64（网络图片地址、图片类型png/jpg、是否需要base64头信息）
     * @param imageUrl
     * @param type
     * @param head
     * @return
     */
    public static String NetImage2Base64(String imageUrl, String type, boolean head) {
        if (StringUtils.isBlank(imageUrl)) {
            return null;
        }

        try {
            URL url = new URL(imageUrl);
            byte[] b = new byte[1024];

            HttpURLConnection conn = (HttpURLConnection) url.openConnection();
            conn.setRequestMethod("GET");
            conn.setConnectTimeout(5000);

            InputStream is = conn.getInputStream();
            ByteArrayOutputStream os = new ByteArrayOutputStream();

            int len = -1;
            while ((len = is.read(b)) != -1) {
                os.write(b, 0, len);
            }

            is.close();

            BASE64Encoder encoder = new BASE64Encoder();
            if (head) {
                if (StringUtils.isBlank(type)) {
                    type = "png";
                }

                return "data:image/" + type + ";base64," + encoder.encode(os.toByteArray());
            } else {
                return encoder.encode(os.toByteArray());
            }
        } catch (Exception e) {
            log.error(e.getMessage());
        }

        return null;
    }

    /**
     * 本地图片转base64（本地图片绝对路径）
     * @param imagePath
     * @return
     */
    public static String LocalImage2Base64(String imagePath) {
        if (StringUtils.isBlank(imagePath)) {
            return null;
        }

        try {
            InputStream is = new FileInputStream(imagePath);
            byte[] b = new byte[is.available()];
            is.read(b);

            is.close();

            BASE64Encoder encoder = new BASE64Encoder();
            return encoder.encode(b);
        } catch (Exception e) {
            log.error(e.getMessage());
        }

        return null;
    }
}
