package top.cenze.utils.file;

import cn.hutool.core.util.StrUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.rendering.PDFRenderer;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;

/**
 * PDF工具类
 */
@Slf4j
public class PDFUtil {
    /**
     * 获取PDF总页数
     * @throws IOException
     */
    public static int getPageNum(String src) throws IOException {
        PDDocument pdDocument = null;
        int pages = 0;

        try {
            pdDocument = getPDDocument(src);
            pages = pdDocument.getNumberOfPages();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.getMessage(), e);
        } finally {
            if (pdDocument != null) {
                pdDocument.close();
            }
        }

        return pages;
    }

    /**
     * PDF转图片 根据页码一页一页转
     * @throws IOException
     */
    public static BufferedImage toImg(String src, int page) throws IOException {
        PDDocument pdDocument = null;
        /* dpi越大转换后越清晰，相对转换速度越慢 */
        int dpi = 100;
        page --;

        try {
            pdDocument = getPDDocument(src);
            PDFRenderer renderer = new PDFRenderer(pdDocument);

            int pages = pdDocument.getNumberOfPages();
            log.info("PdfToImg pages: {}", pages);

            if (page <= pages && page >= 0) {
                BufferedImage image = renderer.renderImageWithDPI(page, dpi);

                return image;
            }
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.getMessage(), e);
        } finally {
            if (pdDocument != null) {
                pdDocument.close();
            }
        }

        return null;
    }

    /**
     * PDF转图片 根据页码一页一页转，写入输出流
     * @throws IOException
     * imgType:转换后的图片类型 jpg,png
     */
    public static void toImgOutputStream(OutputStream os, String src, int page, String imgType) throws IOException {

        BufferedImage bufferedImage = toImg(src, page);

        if (null != bufferedImage) {
            ImageIO.write(bufferedImage, imgType, os);
        }
    }

    private static PDDocument getPDDocument(String src) throws IOException {
        if (StrUtil.isNotEmpty(src)) {
            // http文件，需要使用文件下载链接，否则无法正常读取文件内容（比如使用在线浏览链接，读取则会失败）
            if ("http".equalsIgnoreCase(src.substring(0, 4))) {
                try{

                    URL url= new URL(src);

                    HttpURLConnection connection = (HttpURLConnection) url.openConnection();

                    return PDDocument.load(connection.getInputStream());
                }catch(Exception e) {
                    log.error(e.getMessage(), e);
                }
            }
            // 本地文件
            else {
                File file = new File(src);
                FileInputStream inputStream = new FileInputStream(file);

                return PDDocument.load(inputStream);
            }
        }

        return null;
    }
}
