package top.cenze.utils;

import cn.hutool.core.codec.Base64;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.CharsetUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.crypto.digest.HMac;
import cn.hutool.crypto.digest.HmacAlgorithm;
import lombok.SneakyThrows;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.Header;
import org.apache.http.auth.AuthenticationException;
import top.cenze.utils.crypt.HMacUtil;
import top.cenze.utils.pojo.TokenHeader;
import top.cenze.utils.pojo.TokenPayload;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import javax.servlet.http.HttpServletRequest;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @desc: Token工具
 * @author: chengze
 * @createByDate: 2023/12/13 11:33
 */
public class TokenUtil {

    /**
     * bearer token是一种认证方式，是一种安全令牌，拥有 bearer token 的任何一方（被称为 "bearer"），可以以任何方式，和同样持有它的任何一方一样地使用它来访问受 OAuth 2.0保护的资源。
     * 在java中，一般把bearer token放到请求头的Authorization中，我们可以通过正则来判断token的合法性，并返回头中的有效token
     * https://blog.csdn.net/Gefangenes/article/details/130778194
     * https://www.python100.com/html/90I8A2I7EMG0.html
     * https://blog.csdn.net/weixin_42399342/article/details/112157011
     * @param request
     * @return
     */
    @SneakyThrows
    public static String resolveFromHeaderAuthorizationBearerToken(HttpServletRequest request) {
        String authorization = request.getHeader("Authorization");
        Pattern authorizationPattern = Pattern.compile("^Bearer (?<token>[a-zA-Z0-9-:._~+/]+=*)$", Pattern.CASE_INSENSITIVE);//<token>的值就是真实的表达式配置的值

        if (!StringUtils.startsWithIgnoreCase(authorization, "bearer")) {
            return null;
        }

        Matcher matcher = authorizationPattern.matcher(authorization);
        if (!matcher.matches()) {
            throw new AuthenticationException("Bearer token is malformed");
        }

        // 从上面的正则表达式中获取token
        return matcher.group("token");
    }

    /**
     *
     * https://blog.csdn.net/qq_51705526/article/details/123939826?utm_medium=distribute.pc_relevant.none-task-blog-2~default~baidujs_baidulandingword~default-8-123939826-blog-122104594.235^v39^pc_relevant_3m_sort_dl_base1&spm=1001.2101.3001.4242.5&utm_relevant_index=11
     * https://www.jb51.net/article/259817.htm
     * @param appCode
     * @param appSecret
     * @return
     */
    @SneakyThrows
    public static String createBearerToken(String appCode, String appSecret, Long iat) {
        LinkedHashMap<String, Object> header = new LinkedHashMap<>();
        header.put("alg", "HS256");
        header.put("typ", "JWT");

        if (ObjectUtil.isNull(iat)) {
            iat = System.currentTimeMillis() / 1000;
        }
        LinkedHashMap<String, Object> payload = new LinkedHashMap<>();
        payload.put("iat", iat);
        payload.put("sub", appCode);

        String data = Base64Util.urlEncode(header) + "." + Base64Util.urlEncode(payload);
//        String data = "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJzdWIiOiJjYWlwaSIsImlhdCI6MTUxNjIzOTAyMn0";

//        Mac hmac = Mac.getInstance(HmacAlgorithm.HmacSHA256.getValue());
//        SecretKeySpec secretKey = new SecretKeySpec(appSecret.getBytes("UTF-8"), HmacAlgorithm.HmacSHA256.getValue());
//        hmac.init(secretKey);
//        byte[] bytes = hmac.doFinal(data.getBytes("UTF-8"));
//        return Base64.encodeUrlSafe(bytes);
        return data + "." + HMacUtil.SHA256(data, appSecret);
    }



    public static boolean validBearerToken(String token, String appSecret) {
        if (StrUtil.isEmpty(token) || StrUtil.isEmpty(appSecret)) {
            return false;
        }

        token = token.replaceAll("Bearer ", "");
        if (StrUtil.isEmpty(token)) {
            return false;
        }

        List<String> split = StrUtil.split(token, ".");
        if (CollectionUtil.isEmpty(split) || split.size() < 3) {
            return false;
        }

        String sign = HMacUtil.SHA256(split.get(0) + "." + split.get(1), appSecret);
        return split.get(2).equals(sign);
    }
}
