package top.cenze.utils;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.auth.AuthenticationException;
import top.cenze.utils.crypt.HMacUtil;

import javax.servlet.http.HttpServletRequest;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @desc: Token工具
 * @author: chengze
 * @createByDate: 2023/12/13 11:33
 */
@Slf4j
public class TokenUtil {

    /**
     * 解析BearerToken（HttpServletRequest）
     * bearer token是一种认证方式，是一种安全令牌，拥有 bearer token 的任何一方（被称为 "bearer"），可以以任何方式，和同样持有它的任何一方一样地使用它来访问受 OAuth 2.0保护的资源。
     * 在java中，一般把bearer token放到请求头的Authorization中，我们可以通过正则来判断token的合法性，并返回头中的有效token
     * https://blog.csdn.net/Gefangenes/article/details/130778194
     * https://www.python100.com/html/90I8A2I7EMG0.html
     * https://blog.csdn.net/weixin_42399342/article/details/112157011
     * @param request
     * @return
     */
    public static String resolveBearerTokenFromRequest(HttpServletRequest request) throws AuthenticationException {
        String authorization = request.getHeader("Authorization");

        return resolveBearerTokenFromAuthorization(authorization);
    }

    /**
     * 解析BearerToken（Authorization）
     * @param authorization
     * @return
     * @throws AuthenticationException
     */
    public static String resolveBearerTokenFromAuthorization(String authorization) throws AuthenticationException {
        log.info("resolveBearerTokenFromAuthorization authorization: {}", authorization);
        Pattern authorizationPattern = Pattern.compile("^Bearer (?<token>[a-zA-Z0-9-:._~+/]+=*)$", Pattern.CASE_INSENSITIVE);//<token>的值就是真实的表达式配置的值

        if (!StringUtils.startsWithIgnoreCase(authorization, "bearer")) {
            return null;
        }

        Matcher matcher = authorizationPattern.matcher(authorization);
        if (!matcher.matches()) {
            throw new AuthenticationException("Bearer token is malformed");
        }

        // 从上面的正则表达式中获取token
        return matcher.group("token");
    }

    /**
     * 生成HS256 JWT Token
     * @return
     */
    public static String createHS256JWTToken(String appCode, String appSecret) {
        return createHS256JWTToken(appCode, appSecret, null);
    }

    /**
     * 生成HS256 JWT Token
     * https://blog.csdn.net/qq_51705526/article/details/123939826?utm_medium=distribute.pc_relevant.none-task-blog-2~default~baidujs_baidulandingword~default-8-123939826-blog-122104594.235^v39^pc_relevant_3m_sort_dl_base1&spm=1001.2101.3001.4242.5&utm_relevant_index=11
     * https://www.jb51.net/article/259817.htm
     * @param appId
     * @param appSecret
     * @return
     */
    @SneakyThrows
    public static String createHS256JWTToken(String appId, String appSecret, Long iat) {
        log.info("createHS256JWTToken appId: {}, appSecret: {}, iat: {}", appId, appSecret, iat);
        LinkedHashMap<String, Object> header = new LinkedHashMap<>();
        header.put("alg", "HS256");
        header.put("typ", "JWT");

        if (ObjectUtil.isNull(iat)) {
            iat = System.currentTimeMillis() / 1000;
        }
        LinkedHashMap<String, Object> payload = new LinkedHashMap<>();
        payload.put("iat", iat);
        payload.put("sub", appId);

        String data = Base64Util.urlEncode(header) + "." + Base64Util.urlEncode(payload);
        log.info("createHS256JWTToken data: {}", data);
//        String data = "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJzdWIiOiJjYWlwaSIsImlhdCI6MTUxNjIzOTAyMn0";

//        Mac hmac = Mac.getInstance(HmacAlgorithm.HmacSHA256.getValue());
//        SecretKeySpec secretKey = new SecretKeySpec(appSecret.getBytes("UTF-8"), HmacAlgorithm.HmacSHA256.getValue());
//        hmac.init(secretKey);
//        byte[] bytes = hmac.doFinal(data.getBytes("UTF-8"));
//        return Base64.encodeUrlSafe(bytes);
        return data + "." + HMacUtil.SHA256(data, appSecret);
    }

    /**
     * 转BearerToken
     * @param token
     * @return
     */
    public static String toBearerToken(String token) {
        return "Bearer " + token;
    }

    /**
     * 验证token（鉴权）
     * @param token
     * @param appSecret
     * @return
     */
    public static boolean validBearerToken(String token, String appSecret) {
        log.info("validBearerToken token: {}, appSecret: {}", token, appSecret);
        if (StrUtil.isEmpty(token) || StrUtil.isEmpty(appSecret)) {
            return false;
        }

        token = token.replaceAll("Bearer ", "");
        if (StrUtil.isEmpty(token)) {
            return false;
        }

        List<String> split = StrUtil.split(token, ".");
        if (CollectionUtil.isEmpty(split) || split.size() < 3) {
            return false;
        }

        String sign = HMacUtil.SHA256(split.get(0) + "." + split.get(1), appSecret);
        return split.get(2).equals(sign);
    }
}
