package top.cenze.utils.plugins;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.convert.Convert;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import top.cenze.utils.TokenUtil;
import top.cenze.utils.aspect.AspectUtil;
import top.cenze.utils.pojo.ApiAuthConfig;

import javax.servlet.http.HttpServletRequest;
import java.util.HashSet;
import java.util.Set;

/**
 * @desc: 接口插件
 * @author: chengze
 * @createByDate: 2023/12/30 18:25
 */
@Slf4j
public abstract class ApiPlugin {

    /**
     * 开放接口访问缓存前缀
     */
    private static final String API_ACCESS_PREFIX = "API_ACCESS:";

    /**
     * 接口是否开放的，忽略大小写，返回String（无需登录即可访问，允许三方系统直接访问）
     * @param requestUri    当前请求的uri (ServletPath)
     * @return              返回第一个被请求uri包含的接口uri
     */
    public String isOpenStrIgnoreCase(String requestUri) {
        // 获取开放接口uri集合
        Set<String> setkeys = getOpenApiUris();
        log.info("isOpenStrIgnoreCase openApis: {}", JSON.toJSONString(setkeys));

        // 接口关键字集合为空时，则赋一个空值
        String[] keys = null;
        if (CollectionUtil.isEmpty(setkeys)) {
            setkeys = new HashSet<>();
            setkeys.add("");
            keys = new String[] {""};
        } else {
            keys = Convert.toStrArray(setkeys);
        }

        // 请求uri是否包含接口关键字集合中的成员（忽略大小写），返回集合中uri包含的第一个字符串
        return StrUtil.getContainsStrIgnoreCase(requestUri, keys);
    }

    /**
     * 接口是否开放的，忽略大小写，返回Boolean（无需登录即可访问，允许三方系统直接访问）
     * @param requestUri    当前请求的uri (ServletPath)
     * @return              返回是否为开放接口，true是，false否
     */
    public boolean isOpenIgnoreCase(String requestUri) {
        String containsStr = isOpenStrIgnoreCase(requestUri);
        log.info("isOpenIgnoreCase containsStr: {}", containsStr);

        if (StrUtil.isNotEmpty(containsStr)) {
            return true;
        }

        return false;
    }

    /**
     * 获取开放接口uri列表
     * @return
     */
    public abstract Set<String> getOpenApiUris();

    /**
     * 接口鉴权，返回Boolean
     * @param request   servlet请求
     * @return          返回鉴权结果，通过true，未通过false
     */
    public boolean auth(HttpServletRequest request) {
        return auth(request, false);
    }
    /**
     * 接口鉴权，返回Boolean
     * @param request   servlet请求
     * @param isAop     是否为注解切入
     * @return          返回鉴权结果，通过true，未通过false
     */
    @SneakyThrows
    public boolean auth(HttpServletRequest request, boolean isAop) {
        // 获取系统配置的开放接口中匹配当前请求uri的接口
        String openApiUri = isOpenStrIgnoreCase(request.getServletPath());
        if (StrUtil.isEmpty(openApiUri)) {
            // 如果是注解切入，则鉴权不通过（严格限定）
            if (isAop) {
                return false;
            }
            // 不为注解切入，当前访问非开放接口，则忽略鉴权
            else {
                return true;
            }
        }

        ApiAuthConfig apiAuthConfig = getApiAuthConfig(openApiUri);
        log.info("auth apiAuthConfig: {}", JSON.toJSONString(apiAuthConfig));
        // 鉴权配置为空，或不需要鉴权，则鉴权通过
        if (ObjectUtil.isNull(apiAuthConfig) || !apiAuthConfig.getRequireAuth().equals(1)) {
            return true;
        }

        String token = TokenUtil.resolveBearerTokenFromRequest(request);
        log.info("auth token: {}", token);

        String appSecret = getAuthSecret(token);
        log.info("auth appSecret: {}", appSecret);

        return TokenUtil.validBearerToken(token, appSecret);
    }

    /**
     * 获取token的访问密码
     * @param token
     * @return
     */
    public abstract String getAuthSecret(String token);

    /**
     * 是否超过访问限制次数（配置的限制次数，限制时间范围，单位分钟）
     * @param request       访问请求 servlet请求
     * @return              超过true，未超过false
     */
    public boolean exceedLimitCount(HttpServletRequest request) {
        // 获取系统配置的开放接口中匹配当前请求uri的接口
        String openApiUri = isOpenStrIgnoreCase(request.getServletPath());
        if (StrUtil.isEmpty(openApiUri)) {
            return false;
        }

        String key = API_ACCESS_PREFIX + AspectUtil.getIpAddress(request) + "_" + openApiUri;
        log.info("exceedLimitCount key: {}", key);

        ApiAuthConfig apiAuthConfig = getAccessApiAuthConfig(key);
        log.info("exceedLimitCount ip apiAuthConfig: {}", JSON.toJSONString(apiAuthConfig));
        // ip访问鉴权配置为空，则系统公共配置
        if (ObjectUtil.isNull(apiAuthConfig)) {
            apiAuthConfig = getApiAuthConfig(openApiUri);
            log.info("exceedLimitCount apiAuthConfig: {}", JSON.toJSONString(apiAuthConfig));
        }
        // 鉴权配置为空，则不限制
        if (ObjectUtil.isNull(apiAuthConfig)) {
            return false;
        }

        // 配置的限制次数（0~65535），0不限制
        Integer limitCount = apiAuthConfig.getLimitCount();
        log.info("exceedLimitCount limitCount: {}", limitCount);
        if (ObjectUtil.isNull(limitCount)) {
            limitCount = 0;
        }
        // 配置的限制时间范围（0~ 4294967295，单位：分钟），0不限制（多少分钟内限制访问多少次，limit_count或limit_time为0时，不限制）
        Integer limitTime = apiAuthConfig.getLimitTime();
        log.info("exceedLimitCount limitTime: {}", limitTime);
        if (ObjectUtil.isNull(limitTime)) {
            limitTime = 0;
        }

        // 如果设置了访问次数限制（设置了限制次数及时间范围，且设置值均大于0）
        if (ObjectUtil.isNull(limitCount) || ObjectUtil.isNull(limitCount) ||
                limitCount <= 0 || limitTime <= 0) {
            return false;
        }

        // 当前时间戳，单位：分钟
        Integer curTime = Convert.toInt(System.currentTimeMillis() / 60000);
        log.info("exceedLimitCount curTime: {}", curTime);

        // 访问次数
        Integer accessCount = apiAuthConfig.getAccessCount();
        log.info("exceedLimitCount accessCount: {}", accessCount);
        if (ObjectUtil.isNull(accessCount)) {
            accessCount = 0;
        }
        // 获取当前接口时间范围内初始时间戳，单位：分钟
        Integer accessTime = apiAuthConfig.getAccessFirstTime();
        log.info("exceedLimitCount accessTime: {}", accessTime);
        if (ObjectUtil.isNull(accessTime) || accessTime.equals(0)) {
            accessTime = curTime;
            apiAuthConfig.setAccessFirstTime(curTime);
        }

        // 计算当前限制时间范围内已过去时间，单位分钟
        int pastTime = curTime - accessTime;
        log.info("exceedLimitCount pastTime: {}", pastTime);
        if (pastTime >= limitTime) {
            // 重置访问次数和时间
            apiAuthConfig.setAccessCount(0);
            apiAuthConfig.setAccessFirstTime(0);

            accessCount = 0;
        }

        accessCount ++;
        // 更新缓存访问次数
        apiAuthConfig.setAccessCount(accessCount);
        // 更新缓存
        setAccessApiAuthConfig(key, apiAuthConfig);

        // 超过限制次数
        if (accessCount > limitCount) {
            return true;
        }

        return false;
    }

    /**
     * 获取授权配置
     * @param apiUri
     */
    public abstract ApiAuthConfig getApiAuthConfig(String apiUri);

    /**
     * 获取各ip访问授权配置
     * @param key
     */
    public abstract ApiAuthConfig getAccessApiAuthConfig(String key);

    /**
     * 缓存各ip访问授权配置
     * @param key
     * @param authConfig
     */
    public abstract void setAccessApiAuthConfig(String key, ApiAuthConfig authConfig);
}
