package top.cenze.utils;

import cn.hutool.core.io.FileUtil;
import cn.hutool.core.util.ObjectUtil;
import com.spire.xls.Workbook;
import com.spire.xls.Worksheet;
import com.spire.xls.core.spreadsheet.HTMLOptions;
import lombok.SneakyThrows;
import org.springframework.web.multipart.MultipartFile;
import top.cenze.utils.file.CZFileUtil;
import top.cenze.utils.file.MultipartFileUtil;

import java.io.File;

/**
 * @desc: excel转换工具
 * https://huaweicloud.csdn.net/63875467dacf622b8df8ae8a.html
 * https://repo.e-iceblue.cn/#browse/browse:maven-public:e-iceblue%2Fspire.xls.free%2F2.2.0%2Fspire.xls.free-2.2.0.jar
 * https://blog.csdn.net/zhuocailing3390/article/details/122527100
 * https://gitee.com/lookWord/itext7-demo/tree/master/src/main/java/com/tk/itext7demo/utils/excel
 * https://blog.51cto.com/u_16175446/7287269
 * @author: chengze
 * @createByDate: 2024/1/5 9:19
 */
public class ExcelConvertUtil {
    /**
     * 转PDF（第一个sheet，全部填充至一页内，且列宽根据内容自动调整）
     * @param excelFile
     * @return
     */
    public static MultipartFile toPdf(MultipartFile excelFile) {
        return toPdf(excelFile, 0, true, true);
    }

    /**
     * 转PDF
     * @param excelFile         excel 文件
     * @param sheetNum          第几个sheet（sheet索引，第一个从0开始）
     * @param fitToPage         整个sheet是否都填充到一页PDF内(一页显示所有sheet内容)
     * @param autoFitColumn     sheet列中的字符超过列宽，是否自动调整列宽
     * @return
     */
    @SneakyThrows
    public static MultipartFile toPdf(MultipartFile excelFile, Integer sheetNum, Boolean fitToPage, Boolean autoFitColumn) {
        File inFile = CZFileUtil.loadFile(excelFile);
        if (ObjectUtil.isNull(inFile)) {
            return null;
        }

        MultipartFile tmpFile = null;
        File outFile = CZFileUtil.createPdfFile();
        toPdf(inFile.getPath(), outFile.getPath(), sheetNum, fitToPage, autoFitColumn);
        outFile = CZFileUtil.loadFile(outFile.getPath());
        if (ObjectUtil.isNotNull(outFile) && FileUtil.exist(outFile)) {
            tmpFile = MultipartFileUtil.getMultipartFile(outFile);
        }

        FileUtil.del(inFile);
        FileUtil.del(outFile);

        return tmpFile;
    }

    /**
     * 转PDF
     * @param inputFilePath     excel文件路径
     * @param outputFilePath    输出pdf文件路径
     * @param sheetNum          第几个sheet（sheet索引，第一个从0开始）
     * @param fitToPage         整个sheet是否都填充到一页PDF内(一页显示所有sheet内容)
     * @param autoFitColumn     sheet列中的字符超过列宽，是否自动调整列宽
     */
    public static void toPdf(String inputFilePath, String outputFilePath, Integer sheetNum, Boolean fitToPage, Boolean autoFitColumn) {
        if (ObjectUtil.isNull(sheetNum)) {
            sheetNum = 0;
        }

        Workbook wb = new Workbook();
        // 引入Excel文件
        wb.loadFromFile(inputFilePath);
        Worksheet sheet = wb.getWorksheets().get(sheetNum);
        // 所有内容都填充到一页中
        if (ObjectUtil.isNotNull(fitToPage) && fitToPage) {
            wb.getConverterSetting().setSheetFitToPage(true);
        }
        // 列宽自动调整
        if (ObjectUtil.isNotNull(autoFitColumn) && autoFitColumn) {
            for (int i = 1; i < sheet.getColumns().length; i++)
            {
                sheet.autoFitColumn(i);
            }
        }

        sheet.saveToPdf(outputFilePath);
    }

    /**
     * 转PNG（第一个sheet，全部填充至一页内，且列宽根据内容自动调整）
     * @param excelFile
     * @return
     */
    public static MultipartFile toPng(MultipartFile excelFile) {
        return toPng(excelFile, 0, true, true);
    }

    /**
     * 转PNG
     * @param excelFile         excel 文件
     * @param sheetNum          第几个sheet（sheet索引，第一个从0开始）
     * @param fitToPage         整个sheet是否都填充到一页PNG内(一页显示所有sheet内容)
     * @param autoFitColumn     sheet列中的字符超过列宽，是否自动调整列宽
     * @return
     */
    @SneakyThrows
    public static MultipartFile toPng(MultipartFile excelFile, Integer sheetNum, Boolean fitToPage, Boolean autoFitColumn) {
        File inFile = CZFileUtil.loadFile(excelFile);
        if (ObjectUtil.isNull(inFile)) {
            return null;
        }

        MultipartFile tmpFile = null;
        File outFile = CZFileUtil.createPngFile();
        toPng(inFile.getPath(), outFile.getPath(), sheetNum, fitToPage, autoFitColumn);
        outFile = CZFileUtil.loadFile(outFile.getPath());
        if (ObjectUtil.isNotNull(outFile) && FileUtil.exist(outFile)) {
            tmpFile = MultipartFileUtil.getMultipartFile(outFile);
        }

        FileUtil.del(inFile);
        FileUtil.del(outFile);

        return tmpFile;
    }

    /**
     * 转PNG
     * @param inputFilePath     excel文件路径
     * @param outputFilePath    输出png文件路径
     * @param sheetNum          第几个sheet（sheet索引，第一个从0开始）
     * @param fitToPage         整个sheet是否都填充到一页Png内(一页显示所有sheet内容)
     * @param autoFitColumn     sheet列中的字符超过列宽，是否自动调整列宽
     */
    public static void toPng(String inputFilePath, String outputFilePath, Integer sheetNum, Boolean fitToPage, Boolean autoFitColumn) {
        if (ObjectUtil.isNull(sheetNum)) {
            sheetNum = 0;
        }

        Workbook wb = new Workbook();
        // 引入Excel文件
        wb.loadFromFile(inputFilePath);
        Worksheet sheet = wb.getWorksheets().get(sheetNum);
        // 所有内容都填充到一页中
        if (ObjectUtil.isNotNull(fitToPage) && fitToPage) {
            wb.getConverterSetting().setSheetFitToPage(true);
        }
        // 列宽自动调整
        if (ObjectUtil.isNotNull(autoFitColumn) && autoFitColumn) {
            for (int i = 1; i < sheet.getColumns().length; i++)
            {
                sheet.autoFitColumn(i);
            }
        }

        sheet.saveToImage(outputFilePath);
    }

    /**
     * 转HTML（第一个sheet，全部填充至一页内，且列宽根据内容自动调整）
     * @param excelFile
     * @return
     */
    public static MultipartFile toHtml(MultipartFile excelFile) {
        return toHtml(excelFile, 0, true, true);
    }

    /**
     * 转HTML
     * @param excelFile         excel 文件
     * @param sheetNum          第几个sheet（sheet索引，第一个从0开始）
     * @param fitToPage         整个sheet是否都填充到一页html内(一页显示所有sheet内容)
     * @param autoFitColumn     sheet列中的字符超过列宽，是否自动调整列宽
     * @return
     */
    @SneakyThrows
    public static MultipartFile toHtml(MultipartFile excelFile, Integer sheetNum, Boolean fitToPage, Boolean autoFitColumn) {
        File inFile = CZFileUtil.loadFile(excelFile);
        if (ObjectUtil.isNull(inFile)) {
            return null;
        }

        MultipartFile tmpFile = null;
        File outFile = CZFileUtil.createHtmlFile();
        toHtml(inFile.getPath(), outFile.getPath(), sheetNum, fitToPage, autoFitColumn);
        outFile = CZFileUtil.loadFile(outFile.getPath());
        if (ObjectUtil.isNotNull(outFile) && FileUtil.exist(outFile)) {
            tmpFile = MultipartFileUtil.getMultipartFile(outFile);
        }

//        FileUtil.del(inFile);
//        FileUtil.del(outFile);

        return tmpFile;
    }

    /**
     * 转HTML
     * @param inputFilePath     excel文件路径
     * @param outputFilePath    输出html文件路径
     * @param sheetNum          第几个sheet（sheet索引，第一个从0开始）
     * @param fitToPage         整个sheet是否都填充到一页html内(一页显示所有sheet内容)
     * @param autoFitColumn     sheet列中的字符超过列宽，是否自动调整列宽
     */
    public static void toHtml(String inputFilePath, String outputFilePath, Integer sheetNum, Boolean fitToPage, Boolean autoFitColumn) {
        if (ObjectUtil.isNull(sheetNum)) {
            sheetNum = 0;
        }

        Workbook wb = new Workbook();
        // 引入Excel文件
        wb.loadFromFile(inputFilePath);
        Worksheet sheet = wb.getWorksheets().get(sheetNum);
        // 所有内容都填充到一页中
        if (ObjectUtil.isNotNull(fitToPage) && fitToPage) {
            wb.getConverterSetting().setSheetFitToPage(true);
        }
        // 列宽自动调整
        if (ObjectUtil.isNotNull(autoFitColumn) && autoFitColumn) {
            for (int i = 1; i < sheet.getColumns().length; i++)
            {
                sheet.autoFitColumn(i);
            }
        }

        //Set embedded image as true
        HTMLOptions options = new HTMLOptions();
        options.setImageEmbedded(true);

        sheet.saveToHtml(outputFilePath, options);
    }
}
