package top.cenze.utils;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.convert.Convert;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.io.FileUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.core.util.XmlUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.jexl3.JexlBuilder;
import org.apache.commons.lang.NullArgumentException;
import org.apache.poi.hssf.usermodel.HSSFClientAnchor;
import org.apache.poi.hssf.usermodel.HSSFDateUtil;
import org.apache.poi.hssf.usermodel.HSSFRichTextString;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.xssf.usermodel.*;
import org.jxls.common.Context;
import org.jxls.expression.JexlExpressionEvaluator;
import org.jxls.reader.ReaderBuilder;
import org.jxls.reader.ReaderConfig;
import org.jxls.reader.XLSReadStatus;
import org.jxls.reader.XLSReader;
import org.jxls.transform.Transformer;
import org.jxls.transform.poi.PoiTransformer;
import org.jxls.transform.poi.WritableCellValue;
import org.jxls.transform.poi.WritableHyperlink;
import org.jxls.util.JxlsHelper;
import org.springframework.web.multipart.MultipartFile;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import top.cenze.utils.component.ExcelDateConverter;
import top.cenze.utils.enums.ExcelTypeEnum;
import top.cenze.utils.file.CZFileUtil;
import top.cenze.utils.file.MultipartFileUtil;
import top.cenze.utils.file.XMLUtil;
import top.cenze.utils.pojo.*;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * @desc: JxlsExcel工具类
 * @author: chengze
 * @createByDate: 2023/10/9 10:24
 */
@Slf4j
public class JxlsExcelUtil {

    /** ========================================= delete  ========================================= */

    /**
     * 删除行（从指定起始行开始，删除排除行集合之外的所有行数据）
     * @param file
     * @param sheetIndex
     * @param startRow
     * @param lstExclusionRow
     * @return
     */
    @SneakyThrows
    public static MultipartFile delWithExclusionRow(MultipartFile file,
                                       Integer sheetIndex,
                                       Integer startRow,
                                       List<Integer> lstExclusionRow) {
        if (ObjectUtil.isNull(file) ||
                ObjectUtil.isNull(sheetIndex) ||
                CollectionUtil.isEmpty(lstExclusionRow)) {
            return null;
        }

        Workbook workbook = null;
        if (file.getOriginalFilename().endsWith(".xls")) {
            workbook = new HSSFWorkbook(file.getInputStream());
        }
        if (file.getOriginalFilename().endsWith(".xlsx")) {
            workbook = new XSSFWorkbook(file.getInputStream());
        }

        // worksheet
        Sheet sheet = workbook.getSheetAt(sheetIndex);
        int lastRowNum = sheet.getLastRowNum();

        for (int i = startRow; i < lastRowNum; i++) {
            Row row = sheet.getRow(i);
            if (!lstExclusionRow.contains(i) && ObjectUtil.isNull(row)) {
                sheet.removeRow(row);
            }
        }

        FileItemFactory factory = new DiskFileItemFactory(16, null);
        FileItem fileItem = factory.createItem("textField", "text/plain", true, file.getName());
        OutputStream os = fileItem.getOutputStream();
        workbook.write(os);

        return MultipartFileUtil.getMultipartFile(fileItem);
    }

    /**
     * 删除指定行
     * @param file
     * @param sheetIndex
     * @param lstRow
     * @return
     */
    @SneakyThrows
    public static MultipartFile delRow(MultipartFile file,
                                       Integer sheetIndex,
                                       List<Integer> lstRow) {
        if (ObjectUtil.isNull(file) ||
                ObjectUtil.isNull(sheetIndex) ||
                CollectionUtil.isEmpty(lstRow)) {
            return null;
        }

        Workbook workbook = null;
        if (file.getOriginalFilename().endsWith(".xls")) {
            workbook = new HSSFWorkbook(file.getInputStream());
        }
        if (file.getOriginalFilename().endsWith(".xlsx")) {
            workbook = new XSSFWorkbook(file.getInputStream());
        }

        // worksheet
        Sheet sheet = workbook.getSheetAt(sheetIndex);
//        // 创建样式
//        CellStyle cellStyle = workbook.createCellStyle();
//        cellStyle.setFillForegroundColor(IndexedColors.RED.getIndex());
//        cellStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);

        for (Integer rowIdx : lstRow) {
            Row row = sheet.getRow(rowIdx);
            if (ObjectUtil.isNull(row)) {
                sheet.removeRow(row);
            }
//            for (int i = 0; i < maxCol; i++) {
//                Cell cell = row.getCell(i);
//                cell.setCellStyle(cellStyle);
//            }
        }

        FileItemFactory factory = new DiskFileItemFactory(16, null);
        FileItem fileItem = factory.createItem("textField", "text/plain", true, file.getName());
        OutputStream os = fileItem.getOutputStream();
        workbook.write(os);

        return MultipartFileUtil.getMultipartFile(fileItem);
    }

    /** ========================================= import  ========================================= */

    /**
     * 导入Excel
     * @param file                  待导入数据文件（Excel）
     * @param templateFilePath      导入模板文件路径
     * @param items                 输出对象集合
     * @param skipErrs              是否跳过异常
     * @return
     */
    @SneakyThrows
    public static XLSReadStatus importExcel(MultipartFile file,
                                            String templateFilePath,
                                            Object items,
                                            Boolean skipErrs) {
        // 载入导入模板流
        InputStream isImportXmlTemplate = CZFileUtil.readFile(templateFilePath);

        // 读取待导入数据文件（Excel）
        InputStream isImportFile = file.getInputStream();

        return importExcel(isImportFile, isImportXmlTemplate, items, skipErrs);
    }

    /**
     * 导入Excel
     * @param file                  待导入数据文件（Excel）
     * @param templateFile          导入模板二进制文件（如从数据库中读取的文件）
     * @param items                 输出对象集合
     * @param skipErrs              是否跳过异常
     * @return
     */
    @SneakyThrows
    public static XLSReadStatus importExcel(MultipartFile file,
                                            byte[] templateFile,
                                            Object items,
                                            Boolean skipErrs) {
        // 载入导入模板流
        InputStream isImportXmlTemplate = new ByteArrayInputStream(templateFile);

        // 读取待导入数据文件（Excel）
        InputStream isImportFile = file.getInputStream();

        return importExcel(isImportFile, isImportXmlTemplate, items, skipErrs);
    }

    /**
     * 导入Excel
     * @param importFilePath        待导入数据文件路径（Excel）
     * @param templateFilePath      导入模板文件路径
     * @param items                 输出对象集合
     * @param skipErrs              是否跳过异常
     * @return
     */
    @SneakyThrows
    public static XLSReadStatus importExcel(String importFilePath,
                                            String templateFilePath,
                                            Object items,
                                            Boolean skipErrs) {
        // 载入导入模板流
        InputStream isImportXmlTemplate = CZFileUtil.readFile(templateFilePath);

        // 读取待导入数据文件（Excel）
        InputStream isImportFile = CZFileUtil.readFile(importFilePath);

        return importExcel(isImportFile, isImportXmlTemplate, items, skipErrs);
    }

    /**
     * 导入Excel
     * @param importFilePath    待导入数据文件路径（Excel）
     * @param templateFile      导入模板二进制文件（如从数据库中读取的文件）
     * @param items                 输出对象集合
     * @param skipErrs              是否跳过异常
     * @return
     */
    @SneakyThrows
    public static XLSReadStatus importExcel(String importFilePath,
                                            byte[] templateFile,
                                            Object items,
                                            Boolean skipErrs) {
        // 载入导入模板流
        InputStream isImportXmlTemplate = new ByteArrayInputStream(templateFile);

        // 读取待导入数据文件（Excel）
        InputStream isImportFile = CZFileUtil.readFile(importFilePath);

        return importExcel(isImportFile, isImportXmlTemplate, items, skipErrs);
    }

    /**
     * 导入Excel
     * @param isImportFile          待导入数据文件流（Excel）
     * @param isImportXmlTemplate   载入的导入模板流
     * @param items                 输出对象集合
     * @param skipErrs              是否跳过异常
     * @return
     */
    @SneakyThrows
    public static XLSReadStatus importExcel(InputStream isImportFile,
                                            InputStream isImportXmlTemplate,
                                            Object items,
                                            Boolean skipErrs) {
        // 默认不跳过异常
        if (ObjectUtil.isNull(skipErrs)) {
            skipErrs = false;
        }

        log.info("importExcel items: {}", JSON.toJSONString(items));
        if (ObjectUtil.isNull(isImportFile)) {
            throw new NullArgumentException("待导入数据文件流不能为空");
        }
        if (ObjectUtil.isNull(isImportXmlTemplate)) {
            throw new NullArgumentException("导入模板流不能为空");
        }
        if (ObjectUtil.isNull(items)) {
            throw new NullArgumentException("输出对象集合不能为空");
        }

        if (skipErrs) {
            // 跳过异常
            ReaderConfig.getInstance().setSkipErrors(true);
        }

        // 开启读取偏好设置
        ReaderConfig.getInstance().setUseDefaultValuesForPrimitiveTypes(true);

                // 创建读取器
        XLSReader reader = ReaderBuilder.buildFromXML(isImportXmlTemplate);
        reader.getConvertUtilsBeanProvider().getConvertUtilsBean().register(new ExcelDateConverter(), java.lang.String.class);
//        reader.getConvertUtilsBeanProvider().getConvertUtilsBean().register(new ZonedDateTimeConverter(), java.time.ZonedDateTime.class);
//        reader.getConvertUtilsBeanProvider().getConvertUtilsBean().register(new LocalDateConverter(), java.time.LocalDate.class);
//        reader.getConvertUtilsBeanProvider().getConvertUtilsBean().register(new InstantConverter(), java.time.Instant.class);
//        reader.getConvertUtilsBeanProvider().getConvertUtilsBean().register(new StringConverter(), java.lang.String.class);

        // 创建映射对象集合
        Map beans = new HashMap<>();
        beans.put("items", items);

        return reader.read(isImportFile, beans);
    }

    /**
     * 解释Excel读取映射异常信息
     * @param xlsReadStatus
     * @return
     */
    public static List<ExcelReadException> getReadExceptions(XLSReadStatus xlsReadStatus) {
        // 正则表达式（N个大写字母 + N个数字）
        String regex = "[A-Z]+[0-9]+";
        Pattern pattern = Pattern.compile(regex);

        // 映射失败信息处理
        if (ObjectUtil.isNotNull(xlsReadStatus.getReadMessages())) {
            List<ExcelReadException> lstReadException = new ArrayList<>();
            JSONArray jsonArray = JSONArray.parseArray(JSON.toJSONString(xlsReadStatus.getReadMessages()));
            for (int i = 0; i < jsonArray.size(); i++) {
                JSONObject jsonObject = jsonArray.getJSONObject(i);
                if (ObjectUtil.isNull(jsonObject)) {
                    continue;
                }

                ExcelReadException readException = new ExcelReadException();
                // 映射失败单元格提示
                String errCell = jsonObject.getString("message");
                log.info("getReadExceptions errCell: {}", errCell);
                if (StrUtil.isNotEmpty(errCell)) {
                    // 使用正则表达式提取单元格名称
                    Matcher matcher = pattern.matcher(errCell);
                    while (matcher.find()) {
                        readException.setExCell(matcher.group());
                    }
                }
                JSONObject exception = jsonObject.getJSONObject("exception");
                if (ObjectUtil.isNotNull(exception)) {
                    // 映射失败原因
                    readException.setExMessage(exception.getString("message"));
                }

                lstReadException.add(readException);
            }

            return lstReadException;
        }

        return null;
    }

    /** ========================================= export  ========================================= */

    /**
     * 导出Excel（输出到数据流）
     * @param templateFilePath  Excel模板文件路径
     * @param mapperObject     待导出对象数据（单一对象Object）
     * @param mapperLoop       待导出的循环对象数据（对象Object数组）
     * @param mapperFunc       待载入的自定义对象（用于Excel中引用其对象中的方法）（单一对象Object）
     * @param response
     * @param request
     */
    @SneakyThrows
    public static void exportExcel(String templateFilePath,
                                   Object mapperObject,
                                   Object mapperLoop,
                                   Object mapperFunc,
                                   HttpServletResponse response,
                                   HttpServletRequest request) {
        // 载入模板文件
        File file = new File(templateFilePath);
        FileInputStream fis = new FileInputStream(file);
        // 强转成int类型大小的数组
        byte[] bytes = new byte[(int) file.length()];
        // 将pdf内容放到数组当中
        fis.read(bytes);
        // 关闭文件流
        fis.close();

        exportExcel(bytes, mapperObject, mapperLoop, mapperFunc, response, request);

    }

    /**
     * 导出Excel（输出到数据流）
     * @param templateFile  Excel模板二进制文件（从数据库中读取的）
     * @param mapperObject     待导出对象数据（单一对象Object）
     * @param mapperLoop       待导出的循环对象数据（对象Object数组）
     * @param mapperFunc       待载入的自定义对象（用于Excel中引用其对象中的方法）（单一对象Object）
     * @param response
     * @param request
     */
    @SneakyThrows
    public static void exportExcel(byte[] templateFile,
                                   Object mapperObject,
                                   Object mapperLoop,
                                   Object mapperFunc,
                                   HttpServletResponse response,
                                   HttpServletRequest request) {
        byte[] bytes = exportExcel(templateFile, mapperObject, mapperLoop, mapperFunc);

        ServletOutputStream outputStream = response.getOutputStream();
        outputStream.write(bytes);
        outputStream.flush();
        outputStream.close();
    }

    /**
     * 导出Excel（输出到本地）
     * @param templateFilePath  Excel模板文件路径: xx/xxx/xxx.xlsx
     * @param mapperObject     待导出对象数据（单一对象Object）
     * @param mapperLoop       待导出的循环对象数据（对象Object数组）
     * @param mapperFunc       待载入的自定义对象（用于Excel中引用其对象中的方法）（单一对象Object）
     */
    @SneakyThrows
    public static byte[] exportExcel(String templateFilePath,
                                     Object mapperObject,
                                     Object mapperLoop,
                                     Object mapperFunc) {
        // 载入模板文件
        File file = new File(templateFilePath);
        FileInputStream fis = new FileInputStream(file);
        // 强转成int类型大小的数组
        byte[] bytes = new byte[(int) file.length()];
        // 将pdf内容放到数组当中
        fis.read(bytes);
        // 关闭文件流
        fis.close();

        return exportExcel(bytes, mapperObject, mapperLoop, mapperFunc);
    }

    /**
     * 导出Excel（输出到本地）
     * @param templateFile      Excel模板二进制文件（从数据库中读取的）
     * @param mapperObject     待导出对象数据（单一对象Object）
     * @param mapperLoop       待导出的循环对象数据（对象Object数组）
     * @param mapperFunc       待载入的自定义对象（用于Excel中引用其对象中的方法）（单一对象Object）
     */
    @SneakyThrows
    public static byte[] exportExcel(byte[] templateFile,
                                     Object mapperObject,
                                     Object mapperLoop,
                                     Object mapperFunc) {
        String templateFileName = UUID.randomUUID().toString().replaceAll("-", "") + ExcelTypeEnum.XLSX.getExtname();
        File tmpFile = CZFileUtil.mkFileToResource(null, templateFileName);
        CZFileUtil.writeFileToResource(tmpFile, templateFile);


        String exportFileName = UUID.randomUUID().toString().replaceAll("-", "") + ExcelTypeEnum.XLSX.getExtname();
        File expFile = CZFileUtil.mkFileToResource(null, exportFileName);

        return exportExcel(tmpFile, expFile, mapperObject, mapperLoop, mapperFunc);
    }

    /**
     * 导出Excel
     * @param templateFile          Excel模板文件
     * @param exportFile            输出文件
     * @param mapperObject     待导出对象数据（单一对象Object）
     * @param mapperLoop       待导出的循环对象数据（对象Object数组）
     * @param mapperFunc       待载入的自定义对象（用于Excel中引用其对象中的方法）（单一对象Object）
     */
    @SneakyThrows
    public static byte[] exportExcel(File templateFile,
                                     File exportFile,
                                     Object mapperObject,
                                     Object mapperLoop,
                                     Object mapperFunc) {
        MultipartFile file = exportExcelToMultiPartFile(templateFile, exportFile, mapperObject, mapperLoop, mapperFunc);
        if (ObjectUtil.isNotNull(file)) {
            return file.getBytes();
        }

        return null;
    }

    /**
     * 导出Excel
     * @param templateFile      Excel模板二进制文件（从数据库中读取的）
     * @param mapperObject     待导出对象数据（单一对象Object）
     * @param mapperLoop       待导出的循环对象数据（对象Object数组）
     * @param mapperFunc       待载入的自定义对象（用于Excel中引用其对象中的方法）（单一对象Object）
     */
    @SneakyThrows
    public static MultipartFile exportExcelToMultiPartFile(byte[] templateFile,
                                                           Object mapperObject,
                                                           Object mapperLoop,
                                                           Object mapperFunc) {
        String templateFileName = UUID.randomUUID().toString().replaceAll("-", "") + ExcelTypeEnum.XLSX.getExtname();
        File tmpFile = CZFileUtil.mkFileToResource(null, templateFileName);
        CZFileUtil.writeFileToResource(tmpFile, templateFile);


        String exportFileName = UUID.randomUUID().toString().replaceAll("-", "") + ExcelTypeEnum.XLSX.getExtname();
        File expFile = CZFileUtil.mkFileToResource(null, exportFileName);

        return exportExcelToMultiPartFile(tmpFile, expFile, mapperObject, mapperLoop, mapperFunc);
    }

    /**
     * 导出Excel
     * @param templateFile          Excel模板文件
     * @param exportFile            输出文件
     * @param mapperObject     待导出对象数据（单一对象Object）
     * @param mapperLoop       待导出的循环对象数据（对象Object数组）
     * @param mapperFunc       待载入的自定义对象（用于Excel中引用其对象中的方法）（单一对象Object）
     */
    @SneakyThrows
    public static MultipartFile exportExcelToMultiPartFile(File templateFile,
                                                           File exportFile,
                                                           Object mapperObject,
                                                           Object mapperLoop,
                                                           Object mapperFunc) {
        if (!FileUtil.exist(templateFile) || !FileUtil.exist(exportFile)) {
            throw new Exception("模板或导出文件不存在");
        }

        FileInputStream fis = null;
        FileOutputStream fos = null;
        try {
            // 载入模板文件
            fis = new FileInputStream(templateFile);
            // 创建输出文件流
            fos = new FileOutputStream(exportFile);

            JxlsHelper jxlsHelper = JxlsHelper.getInstance();
            Transformer trans = jxlsHelper.createTransformer(fis, fos);

            // 设置自定义方法（用于Excel模板内引用）
            if (ObjectUtil.isNotNull(mapperFunc)) {
                Map<String , Object> mapFunc = new HashMap<>();
                mapFunc.put("funcs", mapperFunc);
                // 启动新的jxls-api加载自定义方法
                JexlExpressionEvaluator evaluator = (JexlExpressionEvaluator) trans.getTransformationConfig().getExpressionEvaluator();
                evaluator.setJexlEngine(new JexlBuilder().namespaces(mapFunc).create());
            }

            // 创建导出数据
            Context context = PoiTransformer.createInitialContext();
            // 设置导出对象数据
            if (ObjectUtil.isNotNull(mapperObject)) {
                context.putVar("object", mapperObject);
            }
            // 设置导出循环对象数据
            if (ObjectUtil.isNotNull(mapperLoop)) {
                context.putVar("items", mapperLoop);
            }

            //导出
            jxlsHelper.setUseFastFormulaProcessor(false)  // 必须要这个，否者表格函数统计会错乱
                    .processTemplate(context, trans);

            return MultipartFileUtil.getMultipartFile(exportFile);
        } catch (FileNotFoundException fe) {
            log.error("export file not found err: {}", fe.getMessage());
            throw new Exception(fe.getMessage());
        } catch (IOException ie) {
            log.error("export io err: {}", ie.getMessage());
            throw new Exception(ie.getMessage());
        } finally {
            try {
                if (ObjectUtil.isNotNull(fos)) {
                    fos.flush();
                    fos.close();
                }
                if (ObjectUtil.isNotNull(fis)) {
                    fis.close();
                }

                FileUtil.del(templateFile);
                FileUtil.del(exportFile);
            } catch (IOException e) {
                log.error("export io err: {}", e.getMessage());
                throw new Exception(e.getMessage());
            }
        }
    }

    /** ========================================= create export template (xml)  ========================================= */

    /**
     * 给Cell添加批注
     *
     * @param cell 单元格
     * @param value 批注内容
     * @param excelType 扩展名
     */
    private static void addComment(Cell cell, String value, ExcelTypeEnum excelType) {
        if (ObjectUtil.isNull(excelType)) {
            excelType = ExcelTypeEnum.XLSX;
        }

        Sheet sheet = cell.getSheet();
        cell.removeCellComment();
        if (ExcelTypeEnum.XLS.equals(excelType)) {
            ClientAnchor anchor = new HSSFClientAnchor();
            // 关键修改
            anchor.setDx1(0);
            anchor.setDx2(0);
            anchor.setDy1(0);
            anchor.setDy2(0);
            anchor.setCol1(cell.getColumnIndex());
            anchor.setRow1(cell.getRowIndex());
            anchor.setCol2(cell.getColumnIndex());
            anchor.setRow2(cell.getRowIndex());
            // 结束
            Drawing drawing = sheet.createDrawingPatriarch();
            Comment comment = drawing.createCellComment(anchor);
            // 输入批注信息
            comment.setString(new HSSFRichTextString(value));
            // 将批注添加到单元格对象中
            cell.setCellComment(comment);
        } else if (ExcelTypeEnum.XLSX.equals(excelType)) {
            ClientAnchor anchor = new XSSFClientAnchor();
            // 关键修改
            anchor.setDx1(0);
            anchor.setDx2(0);
            anchor.setDy1(0);
            anchor.setDy2(0);
            anchor.setCol1(cell.getColumnIndex());
            anchor.setRow1(cell.getRowIndex());
            anchor.setCol2(cell.getColumnIndex());
            anchor.setRow2(cell.getRowIndex());
            // 结束
            Drawing drawing = sheet.createDrawingPatriarch();
            Comment comment = drawing.createCellComment(anchor);
            // 输入批注信息
            comment.setString(new XSSFRichTextString(value));
            // 将批注添加到单元格对象中
            cell.setCellComment(comment);
        }
    }

    /**
     * 创建导出模板
     * @param mapping
     * @return
     */
    @SneakyThrows
    public static MultipartFile createExportTemplate(ExcelMapping mapping) {
        XSSFWorkbook workbook = new XSSFWorkbook();
        Sheet sheet = workbook.createSheet("Sheet1");
        if (ObjectUtil.isNull(sheet)) {
            throw new NullPointerException("创建sheet失败");
        }

        // 获取所有单元格样式
        Map<Integer, Map<Integer, CellStyle>> mapAllStyle = new HashMap<>();
        Map<Integer, Short> mapAllHeight = new HashMap<>();
        Map<Integer, Integer> mapAllWidth = new HashMap<>();
        getSheetStyle(sheet, mapAllStyle, mapAllHeight, mapAllWidth);

        // 创建所有行
        Map<Integer, Row> allRow = createAllRow(sheet, mapping);
        log.info("createExportTemplate allRow size: {}", allRow.size());
        if (ObjectUtil.isNull(allRow.get(0))) {
            throw new NullPointerException("创建单元格失败");
        }

        // A1单元格
        Cell a1 = allRow.get(0).getCell(0);
        CellStyle a1Style = null;
        if (ObjectUtil.isNull(a1)) {
            allRow.get(0).createCell(0, CellType.STRING);
            a1 = allRow.get(0).getCell(0);
        }
        // 创建所有列及映射
        Map<Integer, Map<Integer, Cell>> allCell = createAllCell(allRow, mapping);

        // 设置A1批注
        String a1Rich = ConvertUtil.numberToLetter(mapping.getMaxCol()) + (mapping.getMaxRow() + 1);
        addComment(a1, "jx:area(lastCell=\"" + a1Rich + "\")", null);
        if (ObjectUtil.isNotNull(a1) && StrUtil.isNotEmpty(a1.getStringCellValue())) {
            a1.setCellValue(a1.getStringCellValue());
        }

        // 设置工作表行高和列宽
        // 读取Luckysheet插件编辑的表格
        setSheetWidthHeight(sheet, mapping.getLstRowHeight(), mapping.getLstColWidth());
        // 读取excel软件编辑的表格
//        setSheetWidthHeight(sheet, mapAllHeight, mapAllWidth);

        // 设置工作表样式
        setSheetStyle(sheet, mapAllStyle);

        // 创建临时模板文件
        File tmpFile = CZFileUtil.createExcelTemplateFile();
        FileOutputStream fos = new FileOutputStream(tmpFile);
        workbook.write(fos);
        workbook.close();
        fos.flush();
        fos.close();

//        // 文件转byte[]
//        byte[] bytes = CZFileUtil.file2Bytes(tmpFile);
        // 文件转multipartFile
        MultipartFile multipartFile = MultipartFileUtil.getMultipartFile(tmpFile);
        // 删除临时模板文件
        FileUtil.del(tmpFile);

        return multipartFile;
    }

    /**
     * 创建导出模板
     * @param templateFile      模板excel文件（主要取其样式，在原样式上生成新的模板）
     * @param mapping           映射数据
     * @return
     */
    @SneakyThrows
    public static MultipartFile createExportTemplate(MultipartFile templateFile, ExcelMapping mapping) {
        log.info("createExportTemplate mapping: {}", JSON.toJSONString(mapping));

        Workbook workbook = null;
        if (templateFile.getOriginalFilename().endsWith(".xls")) {
            workbook = new HSSFWorkbook(templateFile.getInputStream());
        }
        if (templateFile.getOriginalFilename().endsWith(".xlsx")) {
            workbook = new XSSFWorkbook(templateFile.getInputStream());
        }

        // worksheet
        Sheet sheet = workbook.getSheetAt(0);
        if (ObjectUtil.isNull(sheet)) {
            throw new NullPointerException("加载sheet失败");
        }

        // 获取所有单元格样式
        Map<Integer, Map<Integer, CellStyle>> mapAllStyle = new HashMap<>();
        Map<Integer, Short> mapAllHeight = new HashMap<>();
        Map<Integer, Integer> mapAllWidth = new HashMap<>();
        getSheetStyle(sheet, mapAllStyle, mapAllHeight, mapAllWidth);

        // 创建所有行
        Map<Integer, Row> allRow = createAllRow(sheet, mapping);
        log.info("createExportTemplate allRow size: {}", allRow.size());
        if (ObjectUtil.isNull(allRow.get(0))) {
            throw new NullPointerException("创建单元格失败");
        }

        // A1单元格
        Cell a1 = allRow.get(0).getCell(0);
        CellStyle a1Style = null;
        if (ObjectUtil.isNull(a1)) {
            allRow.get(0).createCell(0, CellType.STRING);
            a1 = allRow.get(0).getCell(0);
        }
        // 创建所有列及映射
        Map<Integer, Map<Integer, Cell>> allCell = createAllCell(allRow, mapping);

        // 设置A1批注
        String a1Rich = ConvertUtil.numberToLetter(mapping.getMaxCol()) + (mapping.getMaxRow() + 1);
        addComment(a1, "jx:area(lastCell=\"" + a1Rich + "\")", null);
        if (ObjectUtil.isNotNull(a1) && StrUtil.isNotEmpty(a1.getStringCellValue())) {
            a1.setCellValue(a1.getStringCellValue());
        }

        // 设置工作表行高和列宽
        // 读取Luckysheet插件编辑的表格
        setSheetWidthHeight(sheet, mapping.getLstRowHeight(), mapping.getLstColWidth());
        // 读取excel软件编辑的表格
//        setSheetWidthHeight(sheet, mapAllHeight, mapAllWidth);

        // 设置工作表样式
        setSheetStyle(sheet, mapAllStyle);

        // 创建临时模板文件
        File tmpFile = CZFileUtil.createExcelTemplateFile();
        FileOutputStream fos = new FileOutputStream(tmpFile);
        workbook.write(fos);
        workbook.close();
        fos.flush();
        fos.close();

//        // 文件转byte[]
//        byte[] bytes = CZFileUtil.file2Bytes(tmpFile);
        // 文件转multipartFile
        MultipartFile multipartFile = MultipartFileUtil.getMultipartFile(tmpFile);
        // 删除临时模板文件
        FileUtil.del(tmpFile);

        return multipartFile;
    }

    @SneakyThrows
    public static ExcelMapping getTemplateFileMaxRowAndCol(MultipartFile templateFile) {
        Workbook workbook = null;
        if (templateFile.getOriginalFilename().endsWith(".xls")) {
            workbook = new HSSFWorkbook(templateFile.getInputStream());
        }
        if (templateFile.getOriginalFilename().endsWith(".xlsx")) {
            workbook = new XSSFWorkbook(templateFile.getInputStream());
        }

        // worksheet
        Sheet sheet = workbook.getSheetAt(0);
        if (ObjectUtil.isNull(sheet)) {
            throw new NullPointerException("加载sheet失败");
        }

        int maxRow = sheet.getLastRowNum();
        int maxCol = 0;
        for (int i = 0; i < maxRow + 1; i++) {
            Row row = sheet.getRow(i);
            if (ObjectUtil.isNotNull(row)) {
                short lastCellNum = row.getLastCellNum();
                if (lastCellNum > maxCol) {
                    maxCol = lastCellNum;
                }
            }
        }

        ExcelMapping mapping = new ExcelMapping();
        mapping.setMaxRow(maxRow);
        mapping.setMaxCol(maxCol);

        return mapping;
    }

    /**
     * 获取所有单元格样式
     * @param sheet
     * @return
     */
    private static void getSheetStyle(Sheet sheet,
                                    Map<Integer, Map<Integer, CellStyle>> mapAllStyle,
                                    Map<Integer, Short> mapAllHeight,
                                    Map<Integer, Integer> mapAllWidth) {
        int maxRow = sheet.getLastRowNum();
        int maxCol = 0;
        log.info("getSheetStyle maxRow: {}", maxRow);
        if (maxRow < 0) {
            return;
        }

        // 遍历行
        for (int r = 0; r < maxRow + 1; r++) {
            Row row = sheet.getRow(r);
            // 行高
            if (ObjectUtil.isNull(row)) {
                row = sheet.createRow(r);
            }
            mapAllHeight.put(r, row.getHeight());

            // 获取当前行列数
            int cols = row.getLastCellNum();
            log.info("getSheetStyle cols: {}", cols);
            if (cols > maxCol) {
                maxCol = cols;
            }

            // 遍历列
            Map<Integer, CellStyle> mapStyle = new HashMap<>();
            for (int c = 0; c < cols + 1; c++) {
                Cell cell = row.getCell(c);
                if (ObjectUtil.isNull(cell)) {
                    continue;
                }

                CellStyle cellStyle = cell.getCellStyle();
                if (ObjectUtil.isNotNull(cellStyle)) {
                    mapStyle.put(c, cellStyle);
                }
            }

            // 样式
            if (CollectionUtil.isNotEmpty(mapStyle)) {
                mapAllStyle.put(r, mapStyle);
            }
        }

        log.info("getSheetStyle maxCol: {}", maxCol);
        // 遍历列
        for (int c = 0; c < maxCol + 1; c++) {
            if (ObjectUtil.isNull(sheet.getRow(0).getCell(c))) {
                sheet.getRow(0).createCell(c, CellType.STRING);
            }

            mapAllWidth.put(c, sheet.getColumnWidth(c));
        }

        log.info("getSheetStyle allHeight: {}", JSON.toJSONString(mapAllHeight));
        log.info("getSheetStyle allWidth: {}", JSON.toJSONString(mapAllWidth));
    }

    /**
     * 设置工作表样式
     * @param sheet
     * @param mapAllStyle
     */
    private static void setSheetStyle(Sheet sheet, Map<Integer, Map<Integer, CellStyle>> mapAllStyle) {
        if (ObjectUtil.isNull(sheet) || CollectionUtil.isEmpty(mapAllStyle)) {
            return;
        }

        for (Map.Entry<Integer, Map<Integer, CellStyle>> entry : mapAllStyle.entrySet()) {
            Integer r = entry.getKey();
            Map<Integer, CellStyle> mapStyle = entry.getValue();
            if (CollectionUtil.isEmpty(mapStyle)) {
                continue;
            }

            for (Map.Entry<Integer, CellStyle> e : mapStyle.entrySet()) {
                Integer c = e.getKey();
                CellStyle style = e.getValue();
                if (ObjectUtil.isNull(style)) {
                    continue;
                }

                sheet.getRow(r).getCell(c).setCellStyle(style);
            }
        }
    }

    /**
     * 设置工作表行高和列宽
     * https://zhidao.baidu.com/question/721783003339700525.html
     * @param sheet
     */
    private static void setSheetWidthHeight(Sheet sheet,
                                            Map<Integer, Short> mapAllHeight,
                                            Map<Integer, Integer> mapAllWidth) {
        if (ObjectUtil.isNull(sheet)) {
            return;
        }

        // 设置行高
        if (CollectionUtil.isNotEmpty(mapAllHeight)) {
            for (Map.Entry<Integer, Short> entry : mapAllHeight.entrySet()) {
                if (ObjectUtil.isNull(entry)) {
                    continue;
                }

                // 行高转换计算，可直接等于获取到的列宽值
                // 实际行高 = 获取getWidth / 20
//                BigDecimal h = Convert.toBigDecimal(entry.getValue()).divide(new BigDecimal("20.00"), 3, RoundingMode.HALF_UP);
//                h.multiply(new BigDecimal("20.00")).shortValue();
                sheet.getRow(entry.getKey()).setHeight(entry.getValue());
            }
        }

        // 设置列宽
        if (CollectionUtil.isNotEmpty(mapAllWidth)) {
            for (Map.Entry<Integer, Integer> entry : mapAllWidth.entrySet()) {
                if (ObjectUtil.isNull(entry)) {
                    continue;
                }

                // 列宽转换计算，可直接等于获取到的列宽值
                // 实际列宽 = 获取getHeitht / 264
//                BigDecimal w = Convert.toBigDecimal(entry.getValue()).divide(new BigDecimal("264.00"), 3, RoundingMode.HALF_UP);
//                log.info("setSheetWidthHeight w: {}", w);
//                w.multiply(new BigDecimal("264.00")).intValue();
                sheet.setColumnWidth(entry.getKey(), entry.getValue());
            }
        }
    }

    /**
     * 设置工作表行高和列宽
     * https://zhidao.baidu.com/question/721783003339700525.html
     * @param sheet
     * @param lstRowHeight
     * @param lstColWidth
     */
    private static void setSheetWidthHeight(Sheet sheet, List<Short> lstRowHeight, List<Short> lstColWidth) {
        if (ObjectUtil.isNull(sheet)) {
            return;
        }

        // 设置行高
        if (CollectionUtil.isNotEmpty(lstRowHeight)) {
            short pre = 0;
            for (int r = 0; r < lstRowHeight.size(); r ++) {
                Row row = sheet.getRow(r);
                if (ObjectUtil.isNull(row)) {
                    row = sheet.createRow(r);
                }

                // 实际行高 = 所传高度 * 20
                row.setHeight(Convert.toShort((lstRowHeight.get(r) - pre) * 20));
                pre = lstRowHeight.get(r);
            }
        }

        // 设置列宽
        if (CollectionUtil.isNotEmpty(lstColWidth)) {
            int pre = 0;
            for (int c = 0; c < lstColWidth.size(); c ++) {
                Cell cell = sheet.getRow(0).getCell(c);
                if (ObjectUtil.isNull(cell)) {
                    cell = sheet.getRow(0).createCell(c);
                }

                // 列宽转换计算
                // 实际列宽 = 所传宽度 * 26.4
                sheet.setColumnWidth(c, Convert.toInt((lstColWidth.get(c) - pre) * 26.4));
                pre = lstColWidth.get(c);
            }
        }
    }

    /**
     * 初始化创建所有行
     * @param sheet
     * @param mapping
     * @return
     */
    private static Map<Integer, Row> createAllRow(Sheet sheet, ExcelMapping mapping) {
        if (ObjectUtil.isNull(sheet) || ObjectUtil.isNull(mapping)) {
            return null;
        }

        Map<Integer, Row> mapAllRows = new HashMap<>();
        // 涉及的所有行索引
        Set<Integer> rows = new HashSet<>();

        // 自定义涉及的行索引
        if (CollectionUtil.isNotEmpty(mapping.getLstCustom())) {
            Set<Integer> setRows = mapping.getLstCustom().stream().map(ExcelMappingCustom::getRow).collect(Collectors.toSet());
            if (CollectionUtil.isNotEmpty(setRows)) {
                rows.addAll(setRows);
            }
        }

        // 对象涉及的行索引
        if (CollectionUtil.isNotEmpty(mapping.getLstObject())) {
            Set<Integer> setRows = mapping.getLstObject().stream().map(ExcelMappingObject::getRow).collect(Collectors.toSet());
            if (CollectionUtil.isNotEmpty(setRows)) {
                rows.addAll(setRows);
            }
        }

        // 对象涉及的行索引
        if (ObjectUtil.isNotNull(mapping.getMappingLoop()) && CollectionUtil.isNotEmpty(mapping.getMappingLoop().getLstData())) {
            Set<Integer> setRows = mapping.getMappingLoop().getLstData().stream().map(ExcelMappingObject::getRow).collect(Collectors.toSet());
            if (CollectionUtil.isNotEmpty(setRows)) {
                rows.addAll(setRows);
            }
        }

        // 如果为空或不存在第一行，则添加
        if (CollectionUtil.isEmpty(rows) || !rows.contains(0)) {
            rows.add(0);
        }
        // 获取或创建各行
        for (Integer row : rows) {
            Row excelRow = sheet.getRow(row);
            if (ObjectUtil.isNull(excelRow)) {
                excelRow = sheet.createRow(row);
            }
            if (ObjectUtil.isNotNull(excelRow)) {
                mapAllRows.put(row, excelRow);
            }
        }

        return mapAllRows;
    }

    /**
     * 初始化创建所有列
     * @param mapAllRows
     * @param mapping
     * @return
     */
    private static Map<Integer, Map<Integer, Cell>> createAllCell(Map<Integer, Row> mapAllRows, ExcelMapping mapping) {
        if (CollectionUtil.isEmpty(mapAllRows) || ObjectUtil.isNull(mapping)) {
            return null;
        }

        Map<Integer, Map<Integer, Cell>> mapAllCells = new HashMap<>();

        // 自定义数据（如：标题）
        if (CollectionUtil.isNotEmpty(mapping.getLstCustom())) {
            for (ExcelMappingCustom custom : mapping.getLstCustom()) {
                Row row = mapAllRows.get(custom.getRow());
                if (ObjectUtil.isNull(row)) {
                    throw new NullPointerException("行单元格模板未找到");
                }

                Map<Integer, Cell> mapCells = mapAllCells.get(custom.getRow());
                if (CollectionUtil.isEmpty(mapCells)) {
                    mapCells = new HashMap<>();
                }

                // 单元格内容
                Cell cell = row.getCell(custom.getCol());
                if (ObjectUtil.isNull(cell)) {
                    cell = row.createCell(custom.getCol(), CellType.STRING);
                }

                // 单元格样式
                CellStyle cellStyle = cell.getCellStyle();
                cell.setCellValue(custom.getContent());
                // 设置样式
                if (ObjectUtil.isNotNull(cellStyle)) {
                    cell.setCellStyle(cellStyle);
                }

                mapCells.put(custom.getCol(), cell);
                mapAllCells.put(custom.getRow(), mapCells);
            }
        }

        // 对象数据
        if (CollectionUtil.isNotEmpty(mapping.getLstObject())) {
            for (ExcelMappingObject object : mapping.getLstObject()) {
                Row row = mapAllRows.get(object.getRow());
                if (ObjectUtil.isNull(row)) {
                    throw new NullPointerException("行单元格模板未找到");
                }

                Map<Integer, Cell> mapCells = mapAllCells.get(object.getRow());
                if (CollectionUtil.isEmpty(mapCells)) {
                    mapCells = new HashMap<>();
                }

                StringBuilder sb = new StringBuilder();
                sb.append("${object.").append(object.getObjectItemName()).append("}");

                // 单元格内容
                Cell cell = row.getCell(object.getCol());
                if (ObjectUtil.isNull(cell)) {
                    cell = row.createCell(object.getCol(), CellType.STRING);
                }

                // 单元格样式
                CellStyle cellStyle = cell.getCellStyle();
                cell.setCellValue(sb.toString());
                // 设置样式
                if (ObjectUtil.isNotNull(cellStyle)) {
                    cell.setCellStyle(cellStyle);
                }

                mapCells.put(object.getCol(), cell);
                mapAllCells.put(object.getRow(), mapCells);
            }
        }

        // 循环数据
        if (ObjectUtil.isNotNull(mapping.getMappingLoop()) && CollectionUtil.isNotEmpty(mapping.getMappingLoop().getLstData())) {
            List<ExcelMappingObject> lstData = mapping.getMappingLoop().getLstData();
            Integer minCol = lstData.stream().map(ExcelMappingObject::getCol).min((v1, v2) -> v1 - v2).get();
            Integer maxCol = lstData.stream().map(ExcelMappingObject::getCol).max((v1, v2) -> v1 - v2).get();
            if (ObjectUtil.isNull(minCol)) {
                minCol = 0;
            }

            for (ExcelMappingObject object : lstData) {
//                log.info("createAllCell allRows key: {}, object row: {}", JSON.toJSONString(mapAllRows.keySet()), object.getRow());
                Row row = mapAllRows.get(object.getRow());
                if (ObjectUtil.isNull(row)) {
                    throw new NullPointerException("行单元格模板未找到");
                }

                Map<Integer, Cell> mapCells = mapAllCells.get(object.getRow());
                if (CollectionUtil.isEmpty(mapCells)) {
                    mapCells = new HashMap<>();
                }

                StringBuilder sb = new StringBuilder();
                sb.append("${item.").append(object.getObjectItemName()).append("}");

                // 单元格内容
                Cell cell = row.getCell(object.getCol());
                if (ObjectUtil.isNull(cell)) {
                    cell = row.createCell(object.getCol(), CellType.STRING);
                }

                // 单元格样式
                CellStyle cellStyle = cell.getCellStyle();
                cell.setCellValue(sb.toString());
                // 设置样式
                if (ObjectUtil.isNotNull(cellStyle)) {
                    cell.setCellStyle(cellStyle);
                }

                mapCells.put(object.getCol(), cell);
                mapAllCells.put(object.getRow(), mapCells);
            }

            // jx:each(items="consignees" var="consignee" lastCell="E4")
//            String loopRich = ConvertUtil.numberToLetter(maxCol) + (mapping.getMappingLoop().getMaxRow() + 1);
            // 循环体需占满最大列，否则循环体下面如果有统计数据是超过循环数据最大列的话，超出列的统计数据则会显示在设计时所以的行列位置，并不会显示最后行
            String loopRich = ConvertUtil.numberToLetter(mapping.getMaxCol()) + (mapping.getMappingLoop().getMaxRow() + 1);
            Cell firstLoopCell = mapAllCells.get(mapping.getMappingLoop().getMinRow()).get(minCol);
            if (ObjectUtil.isNull(firstLoopCell)) {
                Row row = mapAllRows.get(mapping.getMappingLoop().getMinRow());
                if (ObjectUtil.isNull(row)) {
                    throw new NullPointerException("行单元格模板未找到");
                }
                firstLoopCell = row.createCell(minCol, CellType.STRING);
            }
            addComment(firstLoopCell, "jx:each(items=\"items\" var=\"item\" lastCell=\"" + loopRich + "\")", null);
        }

        return mapAllCells;
    }

    /** ========================================= create import template (xml)  ========================================= */

    /**
     * 创建导入xml映射文件
     * Excel行或列索引从0开始（即行或列行数据，第一行或第一列，索引为0）
     * @param sheetName         sheet名称
     * @param startRowIndex     真实数据起始行
     * @param lstMapper         数据映射（excel字段名和对应的对应属性名）
     * @param className         映射对象全类型
     * @param
     */
//    public static Document createImportXml(String sheetName, Integer startRowIndex,
//                                            List<ExcelImportMapper> lstMapper, String className) {
//        if (StrUtil.isEmpty(sheetName)) {
//            throw new NullArgumentException("必须指定导入的Excel表Sheet名称");
//        }
//        if (ObjectUtil.isNull(startRowIndex)) {
//            throw new NullArgumentException("必须指定真实数据所在起始行");
//        }
//
//        // 创建xml文档
//        Document xml = XmlUtil.createXml();
//
//        // workbook
//        Element workbook = xml.createElement("workbook");
//
//        // worksheet
//        Element worksheet = xml.createElement("worksheet");
//        if (StrUtil.isNotEmpty(sheetName)) {
//            worksheet.setAttribute("name", sheetName);
//        }
//
//        // 排除（跳过）真实数据起始行前面的行数（如果不跳过非真实数据行，则会出现脏数据，loop即会把非真实数据行读取进来）
//        // exclusionSection
//        Element sectionExclusion = xml.createElement("section");
//        sectionExclusion.setAttribute("startRow", "0");
//        if (startRowIndex > 0) {
//            sectionExclusion.setAttribute("endRow", Convert.toStr(startRowIndex - 1));
//        } else {
//            sectionExclusion.setAttribute("endRow", "0");
//        }
//
//        // loop
//        Element loop = xml.createElement("loop");
//        loop.setAttribute("startRow", Convert.toStr(startRowIndex));
//        loop.setAttribute("endRow", Convert.toStr(startRowIndex));
//        loop.setAttribute("items", "items");
//        loop.setAttribute("var", "item");
//        loop.setAttribute("varType", className);
//
//        // sectionData
//        Element sectionData = xml.createElement("section");
//        sectionData.setAttribute("startRow", Convert.toStr(startRowIndex));
//        sectionData.setAttribute("endRow", Convert.toStr(startRowIndex));
//
//        // mapping
//        for (ExcelImportMapper mapper : lstMapper) {
//            Element mapping = xml.createElement("mapping");
//            mapping.setAttribute("row", Convert.toStr(startRowIndex));
//            mapping.setAttribute("col", Convert.toStr(mapper.getColIndex()));
//            mapping.setTextContent("item." + mapper.getMapperField());
//
//            sectionData.appendChild(mapping);
//        }
//
//        // loopbreakcondition
//        Element loopbreakcondition = xml.createElement("loopbreakcondition");
//
//        // rowcheck
//        Element rowcheck = xml.createElement("rowcheck");
//        rowcheck.setAttribute("offset", "0");
//
//        // cellcheck
//        Element cellcheck = xml.createElement("cellcheck");
//        cellcheck.setAttribute("offset", "0");
//
//        rowcheck.appendChild(cellcheck);
//        loopbreakcondition.appendChild(rowcheck);
//        loop.appendChild(loopbreakcondition);
//        loop.appendChild(sectionData);
//        worksheet.appendChild(sectionExclusion);
//        worksheet.appendChild(loop);
//        workbook.appendChild(worksheet);
//        xml.appendChild(workbook);
//
//        return xml;
//    }

    /**
     * 创建导入模板xml文档
     * @param mapping
     * @return
     */
    @SneakyThrows
    public static MultipartFile createImportXmlToMultipartFile(ExcelMapping mapping) {
        Document xml = createImportXml(mapping);
        if (ObjectUtil.isNull(xml)) {
            return null;
        }

        File tmpFile = XMLUtil.toFile(xml);

        MultipartFile multipartFile = MultipartFileUtil.getMultipartFile(tmpFile);

        // 删除临时文件
        FileUtil.del(tmpFile);

        return multipartFile;
    }

    /**
     * 创建导入模板xml文档
     * @param mapping
     * @return
     */
    public static Document createImportXml(ExcelMapping mapping) {
        if (ObjectUtil.isNull(mapping) || ObjectUtil.isNull(mapping.getMappingLoop())) {
            throw new NullArgumentException("映射内容不能为空");
        }

        // 创建xml文档
        Document xml = XmlUtil.createXml();
        // workbook
        Element workbook = xml.createElement("workbook");

        // 设置工作Sheet索引（默认指定为第一个Sheet）
        if (ObjectUtil.isNull(mapping.getSheetIndex())) {
            mapping.setSheetIndex(0);
        }

        // worksheet
        Element worksheet = xml.createElement("worksheet");
        worksheet.setAttribute("idx", Convert.toStr(mapping.getSheetIndex()));

        // 创建占位元素
        Element exclusionSection = createExclusionSection(xml, 0, mapping.getMappingLoop().getMinRow());
        worksheet.appendChild(exclusionSection);

        // 创建loop
        Element loop = createElementLoop(xml, mapping.getMappingLoop());
        worksheet.appendChild(loop);

        workbook.appendChild(worksheet);
        xml.appendChild(workbook);

        return xml;
    }

    /**
     * 创建Loop元素
     * @param xml
     * @param mappingLoop
     * @return
     */
    private static Element createElementLoop(Document xml, ExcelMappingLoop mappingLoop) {
        // 找出loop的起始行和截止行索引
        Integer startRow = mappingLoop.getMinRow(); // lstLoop.stream().map(ExcelMappingObject::getRow).min((v1, v2) -> v1 - v2).get();
        Integer endRow= mappingLoop.getMaxRow();    // lstLoop.stream().map(ExcelMappingObject::getCol).max((v1, v2) -> v1 - v2).get();
        System.out.println("createElementLoop endRow: " + endRow);

        String varType = mappingLoop.getVarType();
        if (StrUtil.isEmpty(varType)) {
            throw new NullArgumentException("必须指定映射对象的全类名");
        }

        // loop
        Element loop = xml.createElement("loop");
        loop.setAttribute("startRow", Convert.toStr(startRow));
        loop.setAttribute("endRow", Convert.toStr(endRow));
        loop.setAttribute("items", "items");
        loop.setAttribute("var", "item");
        loop.setAttribute("varType", varType);

        // sectionData
        Element sectionData = xml.createElement("section");
        sectionData.setAttribute("startRow", Convert.toStr(startRow));
        sectionData.setAttribute("endRow", Convert.toStr(endRow));

        // mapping
        for (ExcelMappingObject item : mappingLoop.getLstData()) {
            Element mapping = xml.createElement("mapping");
            mapping.setAttribute("row", Convert.toStr(item.getRow()));
            mapping.setAttribute("col", Convert.toStr(item.getCol()));
            mapping.setTextContent("item." + item.getObjectItemName());

            sectionData.appendChild(mapping);
        }

        // loopbreakcondition
        Element loopbreakcondition = xml.createElement("loopbreakcondition");

        // rowcheck
        Element rowcheck = xml.createElement("rowcheck");
        rowcheck.setAttribute("offset", "0");

        // cellcheck
        for (ExcelMappingObject item : mappingLoop.getLstData()) {
            Element cellcheck = xml.createElement("cellcheck");
            cellcheck.setAttribute("offset", Convert.toStr(item.getCol()));

            rowcheck.appendChild(cellcheck);
        }

        loopbreakcondition.appendChild(rowcheck);
        loop.appendChild(loopbreakcondition);
        loop.appendChild(sectionData);

        return loop;
    }

    /**
     * 创建排除（占位）Section元素
     * 排除（跳过）真实数据起始行前面的行数（如果不跳过非真实数据行，则会出现脏数据，loop即会把非真实数据行读取进来）
     * @param xml
     * @param startRow
     * @param endRow
     * @return
     */
    private static Element createExclusionSection(Document xml, Integer startRow, Integer endRow) {
        // exclusionSection
        Element sectionExclusion = xml.createElement("section");
        sectionExclusion.setAttribute("startRow", Convert.toStr(startRow));
        if (ObjectUtil.isNotNull(endRow) && endRow > 0) {
            sectionExclusion.setAttribute("endRow", Convert.toStr(endRow - 1));
        } else {
            sectionExclusion.setAttribute("endRow", "0");
        }

        return sectionExclusion;
    }


    /** ========================================= read  ========================================= */


    /**
     * 读取excel文件指定行的字段信息
     * @param file          excel文件
     * @param sheetIdx      sheet索引（0开始）
     * @param row           字段名所在行数
     * @return
     */
    @SneakyThrows
    public static List<ExcelFieldInfo> readExcelByRow(MultipartFile file, Integer sheetIdx, Integer row) {
        // 默认sheet0（第一个sheet）
        if (ObjectUtil.isNull(sheetIdx)) {
            sheetIdx = 0;
        }
        // 默认第一行
        if (ObjectUtil.isNull(row)) {
            row = 1;
        }

        Workbook workbook = null;
        if (file.getOriginalFilename().endsWith(".xls")) {
            workbook = new HSSFWorkbook(file.getInputStream());
        }
        if (file.getOriginalFilename().endsWith(".xlsx")) {
            workbook = new XSSFWorkbook(file.getInputStream());
        }

        // worksheet
        Sheet sheet = workbook.getSheetAt(sheetIdx);
        int lastRowNum = sheet.getLastRowNum();
        if (lastRowNum < 0) {
            throw new Exception("Excel文件Sheet不能为空");
        }

        // 字段名所在行索引
        int rowIdx = row - 1;
        if (rowIdx > lastRowNum || rowIdx < 0) {
            throw new Exception("指定字段名行数不正确");
        }

        // 获取指定行数据
        Row r = sheet.getRow(rowIdx);
        if (ObjectUtil.isNull(r)) {
            throw new Exception("指定字段名行数据不能为空");
        }

        // 获取指定行列数（总索引总）
        short lastCellNum = r.getLastCellNum();
        if (lastCellNum < 0) {
            throw new Exception("指定字段名行数据列不能为空");
        }

        List<ExcelFieldInfo> lstFields = new ArrayList<>();
        for (int i = 0; i < lastCellNum; i++) {
            Cell cell = r.getCell(i);
            String cellVal = getCellValueByCell(cell);
            if (StrUtil.isEmpty(cellVal)) {
                continue;
            }

            ExcelFieldInfo fieldInfo = new ExcelFieldInfo();
            ConvertUtil.numberToLetter(i);
            fieldInfo.setCell(ConvertUtil.numberToLetter(i) + row);
            fieldInfo.setRowIdx(rowIdx);
            fieldInfo.setColIdx(i);
            fieldInfo.setFieldName(cellVal);

            lstFields.add(fieldInfo);
        }

        return lstFields;
    }

    /**
     * 获取单元格各类型值，返回字符串类型
     *
     * @param cell cell
     * @return String
     */
    public static String getCellValueByCell(Cell cell) {
        //判断是否为null或空串
        if (cell == null || cell.toString().trim().equals("")) {
            return "";
        }

        String cellValue = "";
        CellType cellType = cell.getCellType();
        switch (cellType) {
            case NUMERIC: // 数字
                short format = cell.getCellStyle().getDataFormat();
                if (HSSFDateUtil.isCellDateFormatted(cell)) { // 日期
                    SimpleDateFormat sdf;
                    String dtFormat = "yyyy-MM-dd HH:mm:ss";
                    if (format == 20 || format == 32) {
                        dtFormat = "HH:mm";
                    } else if (format == 14 || format == 31 || format == 57 || format == 58) {
                        dtFormat = "yyyy-MM-dd";
                    } else if (format == 179) {
                        dtFormat = "HH:mm:ss";
                    } else {
                        dtFormat = "yyyy-MM-dd HH:mm:ss";
                    }
                    cellValue = DateUtil.format(cell.getDateCellValue(), dtFormat);
                } else {
                    cell.setCellType(CellType.STRING);
                    cellValue = cell.getStringCellValue();
                }
                break;
            case STRING: // 字符串
                cellValue = cell.getStringCellValue();
                break;
            case BOOLEAN: // Boolean
                cellValue = cell.getBooleanCellValue() + "";
                break;
            case FORMULA: // 公式
                cell.setCellType(CellType.STRING);
                cellValue = cell.getStringCellValue();
                break;
            case BLANK: // 空值
                cellValue = "";
                break;
            case ERROR: // 故障
                cellValue = "ERROR VALUE";
                break;
            default:
                cellValue = "UNKNOWN VALUE";
                break;
        }

        return cellValue;
    }




    /** ========================================= test function  ========================================= */

    /**
     * 字符串转日期（double字符串转Date）
     * @param val
     * @return
     */
    public static String toDateDash(String val) {
        if (StrUtil.isNotEmpty(val)) {
            Double date = Convert.toDouble(val);
            System.out.println("toDate: " + date);

            return DateUtil.format(org.apache.poi.ss.usermodel.DateUtil.getJavaDate(date), "yyyy-MM-dd HH:mm:ss");
        }

        return val;
    }

    /**
     * 字符串转日期（double字符串转Date）
     * @param val
     * @return
     */
    public static String toDate(String val) {
//        System.out.println("toDate: " + val);
        if (StrUtil.isNotEmpty(val)) {
            if (StrUtil.contains(val, "-")) {
//                System.out.println("toDate1: " + val);
                return val;
            } else if (StrUtil.contains(val, "/")) {
//                System.out.println("toDate2: " + val);
                return val.replaceAll("/", "-");
            } else if (StrUtil.contains(val, "\\\\")) {
//                System.out.println("toDate3: " + val);
                return val.replaceAll("\\\\", "-");
            } else {
                Double date = Convert.toDouble(val);
//                System.out.println("toDate4: " + date);

                return doubleToDate(date);
            }
        }

        return val;
    }

    /**
     * double to date
     * @param date
     * @return
     */
    public static String doubleToDate(Double date) {
//        System.out.println("当前时间："+ new Date());
        // Calendar.getInstance() 获取Calendar实例，并获取系统默认的TimeZone
        Calendar calendar = Calendar.getInstance();
//        System.out.println("Calendar的系统默认TimeZone ID：" +
//                calendar.getTimeZone().getID());
//
        // 指定时区，例如意大利的罗马时区：Asia/Shanghai
        // 时区：https://blog.csdn.net/caicai1377/article/details/115873822
        // 罗马时区属于东1区，也就是UTC+1或GMT+1
        TimeZone itTimeZone = TimeZone.getTimeZone("Asia/Shanghai");

        // Calendar指定罗马时区
        calendar.setTimeZone(itTimeZone);
//        System.out.println("Calendar指定TimeZone ID：" + itTimeZone.getID());

        // 夏令时时间，比标准时间快1小时，即3600000毫秒，
        // 根据系统时间计算，如果不在夏令时生效范围内，则为0毫秒，反之为3600000毫秒
        int dstOffset = calendar.get(Calendar.DST_OFFSET);
        // 取得与GMT之间的时间偏移量，例如罗马属于东1区，则时间偏移量为3600000毫秒
        int zoneOffset = calendar.get(Calendar.ZONE_OFFSET);

//        System.out.println("夏令时时间："+dstOffset);
//        System.out.println("时间偏移量："+zoneOffset);
        // 系统时区偏移 1900/1/1 到 1970/1/1 的 25569 天
//        int localOffset = calendar.get(Calendar.ZONE_OFFSET) + calendar.get(Calendar.DST_OFFSET) / (60 * 1000);
//        int localOffset = zoneOffset + dstOffset;
        // 上海时区与GMT时间偏移8小时
        int localOffset = zoneOffset;
//        System.out.println("doubleToDate localOffset: " + localOffset);
        Date tDate = new Date();
        tDate.setTime((long) ((date - 25569) * 24 * 3600 * 1000 - localOffset));

        return DateUtil.format(tDate, "yyyy-MM-dd HH:mm:ss");
    }

    /**
     * 字符串转日期（double字符串转Date）
     * @param val
     * @return
     */
    public static String toDateSlash(String val) {
        if (StrUtil.isNotEmpty(val)) {
            Double date = Convert.toDouble(val);
            System.out.println("toDate: " + date);

            return DateUtil.format(org.apache.poi.ss.usermodel.DateUtil.getJavaDate(date), "yyyy/MM/dd HH:mm:ss");
        }

        return val;
    }

    // 格式化时间（自定义方法，可通过jxls-api加载后，在Excel内引用）
    public Object formatDate(Date dt){
        if(ObjectUtil.isNotNull(dt)){
            String dateStr = DateUtil.format(dt, "yyyy-MM-dd HH:mm:ss");
            return dateStr;
        }

        return "--";
    }

    // 时延-处理执行时间 ms -> s 且保留两位（自定义方法，可通过jxls-api加载后，在Excel内引用）
    public Object timeChange(Long time){
        if(time != null){
            DecimalFormat df = new DecimalFormat("0.00");
            String result = df.format((double)time / 1000);
            return result.equals("0.00") ? "--" : result + "s";
        }
        return "--";
    }

    // 超链接方法（自定义方法，可通过jxls-api加载后，在Excel内引用）
    public WritableCellValue myLink(String address, String title) {
        return new WritableHyperlink(address, title);
    }

    public String getTitle() {
        return "导出收货人";
    }
}
