package top.cenze.utils.file;

import cn.hutool.core.io.FileUtil;
import cn.hutool.core.util.ObjectUtil;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.springframework.util.DigestUtils;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.commons.CommonsMultipartFile;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.OutputStream;

/**
 * ------------------------------------------------------------
 *
 * @Author: chengze
 * @Date: 2021/12/7 18:12
 * @Description: 本地文件转MultipartFile
 * ------------------------------------------------------------
 * 参考文档：https://blog.csdn.net/weixin_43552143/article/details/124212660
 */
@Slf4j
public class MultipartFileUtil {

    // 根据文件绝对路径创建FileItem
    public static FileItem getFileItem(String filePath) {
        File file = new File(filePath);

        return getFileItem(file);
    }

    // 根据文件对象创建FileItem
    public static FileItem getFileItem(File file) {
        log.info("getFileItem file name: {}", file.getName());
        if (ObjectUtil.isNull(file) && !FileUtil.exist(file)) {
            return null;
        }

        FileItemFactory factory = new DiskFileItemFactory(16, null);
        FileItem item = factory.createItem("textField", "text/plain", true, file.getName());
        int bytesRead = 0;
        byte[] buffer = new byte[8192];
        try {
            FileInputStream fis = new FileInputStream(file);
            OutputStream os = item.getOutputStream();
            while ((bytesRead = fis.read(buffer, 0, 8192)) != -1) {
                os.write(buffer, 0, bytesRead);
            }

            os.flush();
            os.close();
            fis.close();
        } catch (IOException e) {
            e.printStackTrace();
        }

        return item;
    }

    // 根据文件绝对路径创建MultipartFile
    public static MultipartFile getMultipartFile(String filePath) {
        FileItem fileItem = getFileItem(filePath);
        if (ObjectUtil.isNull(fileItem)) {
            return null;
        }

        return new CommonsMultipartFile(fileItem);
    }

    // 根据文件对象创建MultipartFile
    public static MultipartFile getMultipartFile(File file) {
        log.info("getMultipartFile file name: {}", file.getName());
        FileItem fileItem = getFileItem(file);
        if (ObjectUtil.isNull(fileItem)) {
            return null;
        }

        return new CommonsMultipartFile(fileItem);
    }

    public static MultipartFile getMultipartFile(FileItem fileItem) {
        if (ObjectUtil.isNull(fileItem)) {
            return null;
        }

        return new CommonsMultipartFile(fileItem);
    }

    /**
     * 获取文件md5码，做为文件的唯一识别码
     * @param file
     * @return
     */
    @SneakyThrows
    public static String getMd5(MultipartFile file) {
        if (ObjectUtil.isNull(file)) {
            return null;
        }

        return DigestUtils.md5DigestAsHex(file.getInputStream());
    }
}
