package top.cenze.utils;

import cn.hutool.core.io.FileUtil;
import cn.hutool.core.util.ObjectUtil;
import com.spire.xls.Workbook;
import com.spire.xls.Worksheet;
import com.spire.xls.core.spreadsheet.HTMLOptions;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.multipart.MultipartFile;
import top.cenze.utils.file.CZFileUtil;
import top.cenze.utils.file.MultipartFileUtil;

import java.io.File;

/**
 * @desc: excel转换工具
 * https://huaweicloud.csdn.net/63875467dacf622b8df8ae8a.html
 * https://repo.e-iceblue.cn/#browse/browse:maven-public:e-iceblue%2Fspire.xls.free%2F2.2.0%2Fspire.xls.free-2.2.0.jar
 * https://blog.csdn.net/zhuocailing3390/article/details/122527100
 * https://gitee.com/lookWord/itext7-demo/tree/master/src/main/java/com/tk/itext7demo/utils/excel
 * https://blog.51cto.com/u_16175446/7287269
 * @author: chengze
 * @createByDate: 2024/1/5 9:19
 */
@Slf4j
public class ExcelConvertUtil {
    /**
     * 转PDF（第一个sheet，全部填充至一页内，且列宽根据内容自动调整）
     * @param excelFile         excel 文件
     * @return
     */
    public static MultipartFile toPdf(MultipartFile excelFile) {
        return toPdf(excelFile, 0, true, true);
    }

    /**
     * 转PDF（第一个sheet，全部填充至一页内，且列宽根据内容自动调整）
     * @param excelFile         excel 文件
     * @param fitToPage         整个sheet是否都填充到一页PDF内(一页显示所有sheet内容)
     * @return
     */
    public static MultipartFile toPdf(MultipartFile excelFile, Boolean fitToPage) {
        return toPdf(excelFile, 0, fitToPage, true);
    }

    /**
     * 转PDF
     * @param excelFile         excel 文件
     * @param sheetNum          第几个sheet（sheet索引，第一个从0开始）
     * @param fitToPage         整个sheet是否都填充到一页PDF内(一页显示所有sheet内容)
     * @param autoFitColumn     sheet列中的字符超过列宽，是否自动调整列宽
     * @return
     */
    @SneakyThrows
    public static MultipartFile toPdf(MultipartFile excelFile, Integer sheetNum, Boolean fitToPage, Boolean autoFitColumn) {
        File inFile = CZFileUtil.loadFile(excelFile);
        log.info("toPdf in file name: {}, length: {}, path: {}", inFile.getName(), inFile.length(), inFile.getPath());
        if (ObjectUtil.isNull(inFile)) {
            return null;
        }

        MultipartFile tmpFile = null;
        File outFile = CZFileUtil.createPdfFile();
        log.info("toPdf out file1 name: {}, length: {}, path: {}", outFile.getName(), outFile.length(), outFile.getPath());
        toPdf(inFile.getPath(), outFile.getPath(), sheetNum, fitToPage, autoFitColumn);
        outFile = CZFileUtil.loadFile(outFile.getPath());
        log.info("toPdf out file2 name: {}, length: {}, path: {}", outFile.getName(), outFile.length(), outFile.getPath());
        if (ObjectUtil.isNotNull(outFile) && FileUtil.exist(outFile)) {
            tmpFile = MultipartFileUtil.getMultipartFile(outFile);
        }

        FileUtil.del(inFile);
        FileUtil.del(outFile);

        log.info("toPdf multi file name: {}, size: {}", tmpFile.getOriginalFilename(), tmpFile.getSize());

        return tmpFile;
    }

    /**
     * 转PDF
     * @param inputFilePath     excel文件路径
     * @param outputFilePath    输出pdf文件路径
     * @param sheetNum          第几个sheet（sheet索引，第一个从0开始）
     * @param fitToPage         整个sheet是否都填充到一页PDF内(一页显示所有sheet内容)
     * @param autoFitColumn     sheet列中的字符超过列宽，是否自动调整列宽
     */
    public static void toPdf(String inputFilePath, String outputFilePath, Integer sheetNum, Boolean fitToPage, Boolean autoFitColumn) {
        if (ObjectUtil.isNull(sheetNum)) {
            sheetNum = 0;
        }

        Workbook wb = new Workbook();
        // 引入Excel文件
        wb.loadFromFile(inputFilePath);
        Worksheet sheet = wb.getWorksheets().get(sheetNum);
        // 所有内容都填充到一页中
        if (ObjectUtil.isNotNull(fitToPage) && fitToPage) {
            wb.getConverterSetting().setSheetFitToPage(true);
        }
        // 列宽自动调整
        if (ObjectUtil.isNotNull(autoFitColumn) && autoFitColumn) {
            for (int i = 1; i < sheet.getColumns().length; i++)
            {
                sheet.autoFitColumn(i);
            }
        }

        sheet.saveToPdf(outputFilePath);
    }

    /**
     * 转PNG（第一个sheet，全部填充至一页内，且列宽根据内容自动调整）
     * @param excelFile
     * @return
     */
    public static MultipartFile toPng(MultipartFile excelFile) {
        return toPng(excelFile, 0, true, true);
    }

    /**
     * 转PNG
     * @param excelFile         excel 文件
     * @param sheetNum          第几个sheet（sheet索引，第一个从0开始）
     * @param fitToPage         整个sheet是否都填充到一页PNG内(一页显示所有sheet内容)
     * @param autoFitColumn     sheet列中的字符超过列宽，是否自动调整列宽
     * @return
     */
    @SneakyThrows
    public static MultipartFile toPng(MultipartFile excelFile, Integer sheetNum, Boolean fitToPage, Boolean autoFitColumn) {
        File inFile = CZFileUtil.loadFile(excelFile);
        if (ObjectUtil.isNull(inFile)) {
            return null;
        }

        MultipartFile tmpFile = null;
        File outFile = CZFileUtil.createPngFile();
        toPng(inFile.getPath(), outFile.getPath(), sheetNum, fitToPage, autoFitColumn);
        outFile = CZFileUtil.loadFile(outFile.getPath());
        if (ObjectUtil.isNotNull(outFile) && FileUtil.exist(outFile)) {
            tmpFile = MultipartFileUtil.getMultipartFile(outFile);
        }

        FileUtil.del(inFile);
        FileUtil.del(outFile);

        return tmpFile;
    }

    /**
     * 转PNG
     * @param inputFilePath     excel文件路径
     * @param outputFilePath    输出png文件路径
     * @param sheetNum          第几个sheet（sheet索引，第一个从0开始）
     * @param fitToPage         整个sheet是否都填充到一页Png内(一页显示所有sheet内容)
     * @param autoFitColumn     sheet列中的字符超过列宽，是否自动调整列宽
     */
    public static void toPng(String inputFilePath, String outputFilePath, Integer sheetNum, Boolean fitToPage, Boolean autoFitColumn) {
        if (ObjectUtil.isNull(sheetNum)) {
            sheetNum = 0;
        }

        Workbook wb = new Workbook();
        // 引入Excel文件
        wb.loadFromFile(inputFilePath);
        Worksheet sheet = wb.getWorksheets().get(sheetNum);
        // 所有内容都填充到一页中
        if (ObjectUtil.isNotNull(fitToPage) && fitToPage) {
            wb.getConverterSetting().setSheetFitToPage(true);
        }
        // 列宽自动调整
        if (ObjectUtil.isNotNull(autoFitColumn) && autoFitColumn) {
            for (int i = 1; i < sheet.getColumns().length; i++)
            {
                sheet.autoFitColumn(i);
            }
        }

        sheet.saveToImage(outputFilePath);
    }

    /**
     * 转HTML（第一个sheet，全部填充至一页内，且列宽根据内容自动调整）
     * @param excelFile
     * @return
     */
    public static MultipartFile toHtml(MultipartFile excelFile) {
        return toHtml(excelFile, 0, true, true);
    }

    /**
     * 转HTML
     * @param excelFile         excel 文件
     * @param sheetNum          第几个sheet（sheet索引，第一个从0开始）
     * @param fitToPage         整个sheet是否都填充到一页html内(一页显示所有sheet内容)
     * @param autoFitColumn     sheet列中的字符超过列宽，是否自动调整列宽
     * @return
     */
    @SneakyThrows
    public static MultipartFile toHtml(MultipartFile excelFile, Integer sheetNum, Boolean fitToPage, Boolean autoFitColumn) {
        File inFile = CZFileUtil.loadFile(excelFile);
        if (ObjectUtil.isNull(inFile)) {
            return null;
        }

        MultipartFile tmpFile = null;
        File outFile = CZFileUtil.createHtmlFile();
        toHtml(inFile.getPath(), outFile.getPath(), sheetNum, fitToPage, autoFitColumn);
        outFile = CZFileUtil.loadFile(outFile.getPath());
        if (ObjectUtil.isNotNull(outFile) && FileUtil.exist(outFile)) {
            tmpFile = MultipartFileUtil.getMultipartFile(outFile);
        }

//        FileUtil.del(inFile);
//        FileUtil.del(outFile);

        return tmpFile;
    }

    /**
     * 转HTML
     * @param inputFilePath     excel文件路径
     * @param outputFilePath    输出html文件路径
     * @param sheetNum          第几个sheet（sheet索引，第一个从0开始）
     * @param fitToPage         整个sheet是否都填充到一页html内(一页显示所有sheet内容)
     * @param autoFitColumn     sheet列中的字符超过列宽，是否自动调整列宽
     */
    public static void toHtml(String inputFilePath, String outputFilePath, Integer sheetNum, Boolean fitToPage, Boolean autoFitColumn) {
        if (ObjectUtil.isNull(sheetNum)) {
            sheetNum = 0;
        }

        Workbook wb = new Workbook();
        // 引入Excel文件
        wb.loadFromFile(inputFilePath);
        Worksheet sheet = wb.getWorksheets().get(sheetNum);
        // 所有内容都填充到一页中
        if (ObjectUtil.isNotNull(fitToPage) && fitToPage) {
            wb.getConverterSetting().setSheetFitToPage(true);
        }
        // 列宽自动调整
        if (ObjectUtil.isNotNull(autoFitColumn) && autoFitColumn) {
            for (int i = 1; i < sheet.getColumns().length; i++)
            {
                sheet.autoFitColumn(i);
            }
        }

        //Set embedded image as true
        HTMLOptions options = new HTMLOptions();
        options.setImageEmbedded(true);

        sheet.saveToHtml(outputFilePath, options);
    }

    @SneakyThrows
    public static void office2PDF(String inputFilePath, String outputFilePath) {
//        // 测试word文档转pdf
//// 创建输入流
//        FileInputStream input = new FileInputStream(inputFilePath);
//// 创建输出流
//        FileOutputStream output = new FileOutputStream(outputFilePath);
//
//
//        LocalOfficeManager.Builder builder = LocalOfficeManager.builder();
//// 设置本地Office地址，推荐LibreOffice
//        builder.officeHome("C:/Program Files/LibreOffice");
//// 部署主机，本地启动
//        builder.hostName("127.0.0.1");
//// 部署端口，可以设置多个
//        builder.portNumbers(9000, 9001, 9002);
//// 单任务过期时间 默认:120000 2分钟
//        builder.taskExecutionTimeout((long) (2 * 1000 * 60));
//// 任务过期时间 默认:30000 3 秒
//        builder.taskQueueTimeout((long) (1000 * 60 * 60));
//// 可以执行的最大任务数，默认200
//        builder.maxTasksPerProcess(100);
//// 构建
//        LocalOfficeManager manager = builder.build();
//// 启动
//        manager.start();
//
//        log.info("office2PDF start");
//        LocalConverter converter = LocalConverter.builder().officeManager(manager).build();
//        // 进行格式转换
//        converter.convert(input).as(DefaultDocumentFormatRegistry.XLSX)
//                .to(output).as(DefaultDocumentFormatRegistry.PDF).execute();
//        log.info("office2PDF end");
//
//        // 关闭流
//        output.close();
//        input.close();
//        manager.stop();
//        DefaultExecutor exec = new DefaultExecutor();
//        File tempFolder = new File(outputFilePath);
//        // 同步等待
//        Semaphore semaphore = new Semaphore(1);
//        semaphore.acquire();
//        ExecuteResultHandler erh = new ExecuteResultHandler() {
//            @Override
//            public void onProcessComplete(int i) {
//                semaphore.release();
//                //转换完成逻辑
//            }
//
//            @Override
//            public void onProcessFailed(ExecuteException e) {
//                semaphore.release();
//                //转换失败逻辑
//                e.printStackTrace();
//            }
//        };
//        String command = "soffice --invisible --convert-to pdf --outdir \"" + tempFolder.getAbsolutePath() + "\" \"" + officeFile.getAbsolutePath() + "\"";
//        System.out.println("执行office文件转换任务，命令为" + command);
//        exec.execute(CommandLine.parse(command), erh);
//        // 等待执行完成
//        semaphore.acquire();
//        File file = new File(tempFolder.getAbsolutePath() + File.separator + officeFile.getName().substring(0, officeFile.getName().indexOf(".")) + ".pdf");
//        if (!file.exists()) {
//            // 转换失败逻辑
//        }
//        return file;
    }
}
