package top.dingwen.io.treasure.async.autoconfigure;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.extra.spring.SpringUtil;
import cn.hutool.json.JSONUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.concurrent.BasicThreadFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Lazy;
import top.dingwen.io.treasure.async.constant.AsyncConstant;
import top.dingwen.io.treasure.async.core.ExecutorManagerImpl;
import top.dingwen.io.treasure.async.core.IExecutorManager;
import top.dingwen.io.treasure.async.core.VisibleThreadPoolTaskExecutor;
import top.dingwen.io.treasure.async.enums.PoolPolicy;
import top.dingwen.io.treasure.async.web.controller.AsyncIndexController;
import top.dingwen.io.treasure.base.util.thread.ThreadUtils;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * 异步处理自动配置类
 *
 * @author dingwen
 * @since 2023/3/5 09:59
 */
@Configuration(proxyBeanMethods = false)
@EnableConfigurationProperties(AsyncProperties.class)
@Lazy(value = false)
@Slf4j
public class AsyncAutoConfiguration {

    @Resource
    private AsyncProperties asyncProperties;

    /**
     * 初始化
     */
    @PostConstruct
    public void init() {
        log.info("{}\n\n" +
                        "                             \n" +
                        "                             \n" +
                        "   __ _ ___ _   _ _ __   ___ \n" +
                        "  / _` / __| | | | '_ \\ / __|\n" +
                        " | (_| \\__ \\ |_| | | | | (__ \n" +
                        "  \\__,_|___/\\__, |_| |_|\\___|\n" +
                        "             __/ |           \n" +
                        "            |___/            ",
                AsyncConstant.AS_LOG_P_S);
        log.info("{},读取到的配置如下:\n{}",
                AsyncConstant.AS_LOG_P_S,
                JSONUtil.toJsonPrettyStr(asyncProperties)
        );
        initExecutor();
    }


    /**
     * 异步场景首页API
     *
     * @return {@link AsyncIndexController }
     */
    @Bean
    @ConditionalOnProperty(name = "dingwen.treasure.async.defaultApiEnabled", havingValue = "true")
    public AsyncIndexController asyncIndexController() {
        log.info("{},{},已经成功加载",AsyncConstant.AS_LOG_P_S,AsyncConstant.AS_LOG_P_AIC);
        return new AsyncIndexController();
    }


    /**
     * 执行器管理组件
     *
     * @return {@link IExecutorManager }
     */
    @Bean
    public IExecutorManager executorManager() {
        log.info("{},{},已经成功加载",AsyncConstant.AS_LOG_P_S,AsyncConstant.AS_LOG_P_EM);
        return new ExecutorManagerImpl();
    }

    /**
     * 初始化线程池
     */
    private void initExecutor() {
        List<PoolProperties> pool = asyncProperties.getPool();
        if (CollUtil.isEmpty(pool)) {
            return;
        }
        pool.forEach(poolProperties -> {
            VisibleThreadPoolTaskExecutor taskExecutor =
                    new VisibleThreadPoolTaskExecutor(poolProperties.getPoolBeanName(), asyncProperties.isLogPrint());
            taskExecutor.setCorePoolSize(poolProperties.getCore());
            taskExecutor.setMaxPoolSize(poolProperties.getMax());
            taskExecutor.setQueueCapacity(poolProperties.getQueueCapacity());
            taskExecutor.setKeepAliveSeconds(poolProperties.getKeepAliveTime());
            taskExecutor.setThreadNamePrefix(poolProperties.getPoolNamePrefix());
            taskExecutor.setRejectedExecutionHandler(adapterPoolPolicy(poolProperties.getPoolPolicy()));
            taskExecutor.initialize();
            SpringUtil.registerBean(poolProperties.getPoolBeanName(), taskExecutor);
            log.info("{},线程池初始化,线程池名称:{},前缀:{},核心线程数:{},最大线程数:{},缓存队列:{},存活时间:{},拒绝策略:{}",
                   AsyncConstant.AS_LOG_P_S,
                    poolProperties.getPoolBeanName(),
                    poolProperties.getPoolNamePrefix(),
                    poolProperties.getCore(),
                    poolProperties.getMax(),
                    poolProperties.getQueueCapacity(),
                    poolProperties.getKeepAliveTime(),
                    poolProperties.getPoolPolicy());

        });
        log.info("{},启动成功，总计初始化线程池{}个",AsyncConstant.AS_LOG_P_S, pool.size());
    }


    /**
     * 适配器线程池拒接策略
     *
     * @param poolPolicy 池政策
     * @return {@link RejectedExecutionHandler}
     */
    private RejectedExecutionHandler adapterPoolPolicy(PoolPolicy poolPolicy) {
        if (Objects.isNull(poolPolicy)) {
            return new ThreadPoolExecutor.CallerRunsPolicy();
        }
        switch (poolPolicy) {
            case ABORT_POLICY:
                return new ThreadPoolExecutor.AbortPolicy();
            case DISCARD_POLICY:
                return new ThreadPoolExecutor.DiscardPolicy();
            case DISCARD_OLDEST_POLICY:
                return new ThreadPoolExecutor.DiscardOldestPolicy();
            default:
                return new ThreadPoolExecutor.CallerRunsPolicy();
        }
    }

    /**
     *定时线程池
     *
     * @return {@link ScheduledExecutorService }
     */
    @ConditionalOnProperty(name = "dingwen.treasure.async.delayScheduledPollEnabled", havingValue = "true")
    @Bean
    public ScheduledExecutorService defaultScheduledExecutorService() {
        return new ScheduledThreadPoolExecutor(asyncProperties.getDelayScheduledPollCoreSize(),
                new BasicThreadFactory.Builder().namingPattern("async-default-schedule-pool-%d").daemon(true).build()) {
            @Override
            protected void afterExecute(Runnable r, Throwable t) {
                super.afterExecute(r, t);
                ThreadUtils.printException(r, t);
            }
        };
    }
}
