package top.lingkang.finalsecurity.config;

import jakarta.servlet.*;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import top.lingkang.finalsecurity.common.base.FinalAuth;
import top.lingkang.finalsecurity.common.base.FinalHttpProperties;
import top.lingkang.finalsecurity.common.base.FinalSessionObject;
import top.lingkang.finalsecurity.common.error.FinalBaseException;
import top.lingkang.finalsecurity.common.error.FinalNotLoginException;
import top.lingkang.finalsecurity.common.error.FinalPermissionException;
import top.lingkang.finalsecurity.common.utils.AuthUtils;
import top.lingkang.finalsecurity.http.FinalRequestContext;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author lingkang
 * Created by 2022/2/11
 * 过滤配置类（核心）
 * 匹配优先级别：排除 > 鉴权
 * @since 1.0.0
 */
public class FinalSecurityConfiguration implements Filter {
    private FinalHttpProperties properties = new FinalHttpProperties();
    private FinalSessionObject sessionObject = new FinalSessionObjectServlet();

    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain chain) throws IOException, ServletException {
        HttpServletRequest request = (HttpServletRequest) servletRequest;
        FinalRequestContext.setRequest(request);
        try {
            String path = request.getServletPath();
            // 缓存相关
            if (properties.getCheckPathCache().getExcludePath().contains(path)) {
                chain.doFilter(servletRequest, servletResponse);
                return;
            } else if (properties.getCheckPathCache().getAuths().containsKey(path)) {
                sessionObject.init(request.getSession());
                FinalAuth[] finalAuths = properties.getCheckPathCache().getAuths().get(path);
                for (FinalAuth auth : finalAuths) {
                    auth.check(sessionObject);
                }

                chain.doFilter(servletRequest, servletResponse);
                return;
            }

            // 排除
            for (String url : properties.getExcludePath()) {
                if (AuthUtils.matcher(url, path)) {
                    properties.getCheckPathCache().getExcludePath().add(path);// 添加缓存
                    chain.doFilter(servletRequest, servletResponse);
                    return;
                }
            }

            // 检查角色
            HashMap<String, FinalAuth> checkAuth = properties.getCheckAuthorize().getAuthorize();
            List<FinalAuth> auths = new ArrayList<>();
            for (Map.Entry<String, FinalAuth> entry : checkAuth.entrySet()) {
                if (AuthUtils.matcher(entry.getKey(), path)) {
                    auths.add(entry.getValue());
                }
            }

            // cache
            properties.getCheckPathCache().getAuths().put(path, AuthUtils.AllToOne(auths.toArray(new FinalAuth[auths.size()])));

            // 不需要校验
            if (auths.isEmpty()) {
                chain.doFilter(servletRequest, servletResponse);
                return;
            }

            // 执行检查
            sessionObject.init(request.getSession());
            for (FinalAuth auth : auths) {
                auth.check(sessionObject);
            }

            //放行
            chain.doFilter(servletRequest, servletResponse);
        } catch (Exception e) {
            if (FinalBaseException.class.isAssignableFrom(e.getClass()) || FinalBaseException.class.isAssignableFrom(e.getCause().getClass())) {
                if (e instanceof FinalPermissionException) {
                    properties.getExceptionHandler().permissionException(e, request, (HttpServletResponse) servletResponse);
                } else if (e.getCause() instanceof FinalPermissionException)
                    properties.getExceptionHandler().permissionException(e.getCause(), request, (HttpServletResponse) servletResponse);
                else if (e instanceof FinalNotLoginException) {
                    properties.getExceptionHandler().notLoginException(e, request, (HttpServletResponse) servletResponse);
                } else if (e.getCause() instanceof FinalNotLoginException)
                    properties.getExceptionHandler().notLoginException(e.getCause(), request, (HttpServletResponse) servletResponse);
                else
                    properties.getExceptionHandler().exception(e, request, (HttpServletResponse) servletResponse);
            } else {
                throw e;
            }
        } finally {
            FinalRequestContext.removeRequest();
        }
    }

    protected void config(FinalHttpProperties properties) {
        this.properties = properties;
    }

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {
        if (properties.getExceptionHandler() == null)
            properties.setExceptionHandler(new DefaultFinalExceptionHandler());

        config(properties);
    }

    public FinalHttpProperties getProperties() {
        return properties;
    }
}
