package top.lingkang.finalsql.sql;


import top.lingkang.finalsql.annotation.Id;
import top.lingkang.finalsql.config.SqlConfig;
import top.lingkang.finalsql.constants.IdType;
import top.lingkang.finalsql.constants.JdbcVersion;
import top.lingkang.finalsql.error.ResultHandlerException;
import top.lingkang.finalsql.utils.AssertUtils;
import top.lingkang.finalsql.utils.ClassUtils;
import top.lingkang.finalsql.utils.NameUtils;

import java.lang.reflect.Field;
import java.sql.Blob;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author lingkang
 * Created by 2022/4/11
 */
public class ResultHandler {
    private SqlConfig sqlConfig;

    public ResultHandler(SqlConfig config) {
        this.sqlConfig = config;
    }

    public <T> List<T> list(ResultSet resultSet, Class<T> clazz) {
        AssertUtils.notNull(resultSet, "ResultSet 结果集不能为空！");
        try {
            List<T> list = new ArrayList<>();
            //获取要封装的javabean声明的属性
            Field[] fields = ClassUtils.getColumnField(clazz);
            //遍历ResultSet
            while (resultSet.next()) {
                T obj = (T) clazz.newInstance();
                //匹配JavaBean的属性,然后赋值
                for (Field field : fields) {
                    field.setAccessible(true);

                    if (sqlConfig.getJdbcVersion() == JdbcVersion.JDBC_V4)
                        field.set(obj, resultSet.getObject(field.getName(), field.getType()));
                    else
                        field.set(obj, resultSet.getObject(field.getName()));
                }
                list.add(obj);
            }
            return list;
        } catch (SQLException | IllegalAccessException | InstantiationException e) {
            throw new ResultHandlerException(e);
        }
    }

    public <T> List<T> selectForList(ResultSet result, Class<T> clazz) throws Exception {
        List<T> list = new ArrayList<>();
        for (; ; ) {
            T t = selectForObject(result, clazz);
            if (t == null)
                break;
            list.add(t);
        }
        return list;
    }

    public <T> T selectForObject(ResultSet result, Class<T> entity) throws Exception {
        if (entity == Map.class) {
            if (result.next())
                return (T) selectForMap(result);
        } else if (ClassUtils.isBaseWrapper(entity)) {
            if (result.next()) {
                if (sqlConfig.getJdbcVersion() == JdbcVersion.JDBC_V4)
                    return result.getObject(1, entity);
                else
                    return (T) result.getObject(1);
            }

        } else if (entity == Blob.class) {
            if (result.next())
                return (T) result.getBlob(1);
        } else {
            if (result.next()) {
                ResultSetMetaData metaData = result.getMetaData();
                T ins = entity.newInstance();// 实例化对象
                for (int i = 1; i <= metaData.getColumnCount(); i++) {

                    Field field = ClassUtils.getField(
                            NameUtils.toHump(metaData.getColumnLabel(i)),
                            ins.getClass().getDeclaredFields()
                    );
                    if (field != null) {
                        field.setAccessible(true);

                        if (sqlConfig.getJdbcVersion() == JdbcVersion.JDBC_V4)
                            field.set(ins, result.getObject(i, field.getType()));
                        else
                            field.set(ins, result.getObject(i));
                    }
                }
                return ins;
            }
        }
        return null;
    }


    public <T> T oneResult(ResultSet resultSet, Class<T> clazz) {
        try {
            //获取要封装的javabean声明的属性
            Field[] fields = ClassUtils.getColumnField(clazz);
            T entity = clazz.newInstance();// 实例化

            //匹配JavaBean的属性,然后赋值
            for (Field field : fields) {
                field.setAccessible(true);

                if (sqlConfig.getJdbcVersion() == JdbcVersion.JDBC_V4)
                    field.set(entity, resultSet.getObject(field.getName(), field.getType()));
                else
                    field.set(entity, resultSet.getObject(field.getName()));
            }
            return entity;
        } catch (SQLException | IllegalAccessException | InstantiationException e) {
            throw new ResultHandlerException(e);
        }
    }

    public <T> int insert(ResultSet resultSet, T entity) throws SQLException, IllegalAccessException {
        Id id = ClassUtils.getId(entity.getClass());
        if (id != null && id.value() == IdType.INPUT) {
            return 1;
        }
        Class<?> clazz = entity.getClass();
        Field idColumn = ClassUtils.getIdColumn(clazz.getDeclaredFields());
        if (idColumn != null) {
            idColumn.setAccessible(true);

            if (sqlConfig.getJdbcVersion() == JdbcVersion.JDBC_V4)
                idColumn.set(entity, resultSet.getObject(1, idColumn.getType()));
            else
                idColumn.set(entity, resultSet.getObject(1));
        }
        return 1;
    }

    /**
     * @param resultSet
     * @param entity    不为空
     * @param <T>
     * @return
     * @throws SQLException
     * @throws IllegalAccessException
     */
    public <T> int batchInsert(ResultSet resultSet, List<T> entity) throws SQLException, IllegalAccessException {
        boolean notSetId = false;
        Id id = ClassUtils.getId(entity.get(0).getClass());
        if (id != null && id.value() == IdType.INPUT) {
            notSetId = true;
        }

        int i = 0;
        while (resultSet.next()) {
            if (notSetId) {// 不设置返回 id，适用常见数据库
                continue;
            }
            Class<?> clazz = entity.get(i).getClass();
            Field idColumn = ClassUtils.getIdColumn(clazz.getDeclaredFields());
            if (idColumn != null) {
                idColumn.setAccessible(true);
                if (sqlConfig.getJdbcVersion() == JdbcVersion.JDBC_V4)
                    idColumn.set(entity.get(i), resultSet.getObject(1, idColumn.getType()));
                else
                    idColumn.set(entity.get(i), resultSet.getObject(1));
            }
            i++;
        }
        return i;
    }

    public Map selectForMap(ResultSet result) throws SQLException {
        Map<String, Object> map = new HashMap<>();
        ResultSetMetaData metaData = result.getMetaData();
        for (int i = 1; i <= metaData.getColumnCount(); i++) {
            map.put(NameUtils.toHump(metaData.getColumnLabel(i)), result.getObject(i));
        }
        return map;
    }

}
