package top.lingkang.finalvalidated.utils;

import java.lang.reflect.Method;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author lingkang
 * created by 2025-10-13
 */
class GetterUtils {
    protected static Map<String, Method> CACHE_METHOD = new ConcurrentHashMap<>();

    /**
     * 获取属性值（自动类型转换）
     */
    @SuppressWarnings("unchecked")
    public static <T> T getValue(Object obj, String propertyName) {
        Object value = getValueRaw(obj, propertyName);
        return (T) value;
    }

    /**
     * 获取原始值
     */
    public static Object getValueRaw(Object obj, String propertyName) {
        if (obj == null) {
            throw new IllegalArgumentException("对象不能为null");
        }
        if (propertyName == null || propertyName.length() == 0) {
            throw new IllegalArgumentException("属性名不能为空");
        }

        try {
            Method getter = CACHE_METHOD.get(obj.getClass().getName() + propertyName);
            if (getter == null) {
                getter = findGetterMethod(obj.getClass(), propertyName);
                // 缓存
                CACHE_METHOD.put(obj.getClass().getName() + propertyName, getter);
            }
            return getter.invoke(obj);
        } catch (Exception e) {
            throw new RuntimeException("获取属性值失败: " + e.getMessage(), e);
        }
    }

    /**
     * 安全获取值，如果失败返回默认值
     */
    public static <T> T getValueSafe(Object obj, String propertyName, T defaultValue) {
        try {
            return getValue(obj, propertyName);
        } catch (Exception e) {
            return defaultValue;
        }
    }

    /**
     * 安全获取值，如果失败返回null
     */
    public static <T> T getValueSafe(Object obj, String propertyName) {
        return getValueSafe(obj, propertyName, null);
    }

    /**
     * 查找 getter 方法
     */
    private static Method findGetterMethod(Class<?> clazz, String propertyName) {
        String[] possibleMethods = {
                "get" + capitalize(propertyName),  // getXxx
                "is" + capitalize(propertyName)    // isXxx
        };

        for (String methodName : possibleMethods) {
            try {
                return clazz.getMethod(methodName);
            } catch (NoSuchMethodException e) {
                // 继续尝试下一个可能的方法名
            }
        }

        throw new RuntimeException(
                "在类 " + clazz.getName() + " 中未找到属性 '" + propertyName + "' 的 getter 方法");
    }

    private static String capitalize(String str) {
        if (str == null || str.isEmpty()) return str;
        return Character.toUpperCase(str.charAt(0)) + str.substring(1);
    }
}
