package top.lingkang.mm.orm;

import org.apache.ibatis.annotations.*;
import top.lingkang.mm.annotation.MagicMapper;

import java.util.List;

/**
 * mapper 的基础增删查改
 *
 * @author lingkang
 * Created by 2024/3/11
 */
@Mapper
@MagicMapper
public interface BaseMapper<T> {
    /**
     * 查询所有
     */
    @Select(BaseMapperSql.selectAll)
    @Lang(BaseMapperDriver.class)
    List<T> selectAll();

    /**
     * 根据id查询
     *
     * @param id 主键id
     * @return 实体对象
     */
    @Select(BaseMapperSql.selectById)
    @Lang(BaseMapperDriver.class)
    T selectById(@Param(BaseMapperSql.param_id) Object id);

    /**
     * 查询表数据总行数
     *
     * @return 总行数
     */
    @Select(BaseMapperSql.selectCount)
    @Lang(BaseMapperDriver.class)
    long selectCount();

    /**
     * 查询指定列，并返回指定结果，例如：
     * <pre>
     * {@code
     * UserMapper userMapper = mapperManage.getMapper(UserMapper.class);
     * List<Long> list = userMapper.selectColumn(
     *         new QueryColumn(Long.class, "id")
     *                 .gt("id", 1) // 大于1 的id列
     * );
     * }
     * </pre>
     *
     * @param query 查询条件，比较符可以参考：{@link Condition}
     * @param <E>   结果对象类
     * @return 结果列表
     */
    @Select(BaseMapperSql.selectColumn)
    @Lang(BaseMapperDriver.class)
    <E> List<E> selectColumn(@Param(BaseMapperSql.param_q2) QueryColumn query);

    /**
     * 查询指定列，并返回一行结果，若返回多个结果时，将会报错
     * <pre>
     * {@code
     *     Long result = userMapper.selectColumnOne(
     *                 new QueryColumn(Long.class, "id").sql("limit 1"));
     * }
     * </pre>
     *
     * @param query 查询条件，比较符可以参考：{@link Condition}
     * @param <E>   结果对象类
     * @return 结果对象，结果是多行时抛出异常：{@link org.apache.ibatis.exceptions.TooManyResultsException}
     */
    @Select(BaseMapperSql.selectColumnOne)
    @Lang(BaseMapperDriver.class)
    <E> E selectColumnOne(@Param(BaseMapperSql.param_q2) QueryColumn query);

    /**
     * 判断是否存在
     *
     * @param id 主键id
     * @return true：存在；false：不存在
     */
    @Select(BaseMapperSql.existsById)
    @Lang(BaseMapperDriver.class)
    boolean existsById(@Param(BaseMapperSql.param_id) Object id);

    /**
     * 创建查询，例如：
     * <pre>
     * {@code
     * // 查询id为 1 的实体对象
     * List<UserEntity> list = mapper.createQuery(new Query().eq("id", 1))
     * }
     * </pre>
     *
     * @param query 查询条件，比较符可以参考：{@link Condition}
     * @return 结果列表
     */
    @Select(BaseMapperSql.createQuery)
    @Lang(BaseMapperDriver.class)
    List<T> createQuery(@Param(BaseMapperSql.param_q) Query query);

    /**
     * 插入数据
     *
     * @param entity 实体对象
     * @return 受影响的行数
     */
    @Insert(BaseMapperSql.insert)
    @Lang(BaseMapperDriver.class)
    @Options(useGeneratedKeys = true)
    int insert(@Param(BaseMapperSql.magic_base_e) T entity);

    /**
     * 批量插入数据，例如
     * <pre>
     * {@code
     *         List<UserEntity> list = new ArrayList<>();
     *         UserEntity user = new UserEntity();
     *         user.setId(System.currentTimeMillis());
     *         UserEntity user1 = new UserEntity();
     *         user1.setId(System.currentTimeMillis() + 1);
     *         list.add(user);
     *         list.add(user1);
     *         UserMapper mapper = mapperManage.getMapper(UserMapper.class);
     *         int insertBatch = mapper.insertBatch(list);
     *         log.info("insertBatch list: {}", list);
     *         log.info("insertBatch {}", insertBatch);
     * }
     * </pre>
     *
     * @param list 实体对象列表
     * @return 受影响的行数
     */
    @Insert(BaseMapperSql.insertBatch)
    @Lang(BaseMapperDriver.class)
    @Options(useGeneratedKeys = true)
    int insertBatch(@Param(BaseMapperSql.magic_base_list) List<T> list);

    /**
     * 根据id进行更新
     *
     * @param entity 实体对象
     * @return 受影响的行数
     */
    @Update(BaseMapperSql.updateById)
    @Lang(BaseMapperDriver.class)
    int updateById(@Param(BaseMapperSql.magic_base_e) T entity);

    /**
     * 根据主键id删除
     *
     * @param id 主键id、实体对象类
     * @return 受影响的行数
     */
    @Delete(BaseMapperSql.deleteById)
    @Lang(BaseMapperDriver.class)
    int deleteById(@Param(BaseMapperSql.param_id) Object id);
}
