package top.lshaci.framework.common.exception;

import cn.hutool.core.util.StrUtil;
import lombok.Getter;
import top.lshaci.framework.common.constants.ErrorInfo;

/**
 * <p>Framework common base exception</p>
 *
 * <p>1.0.4:</p>
 * <pre>
 *     1. 添加异常码（code）字段
 * </pre>
 *
 * <p>1.1.0:</p>
 * <pre>
 *     1. 添加异常信息（errorInfo）字段
 *     2. 重写 toString 方法
 * </pre>
 *
 * @author lshaci
 * @version 1.1.0
 * @since 0.0.1
 */
public class BaseException extends RuntimeException {

    private static final long serialVersionUID = -8170023045088441807L;

    /**
     * 异常码
     */
    @Getter
    private final int code;

    /**
     * 异常信息
     */
    @Getter
    private ErrorInfo errorInfo;

    /**
     * 根据 异常信息 创建一个基础异常
     *
     * @param code     异常码
     * @param template 异常消息
     * @param args     消息参数
     */
    public BaseException(int code, String template, Object... args) {
        super(StrUtil.format(template, args));
        this.code = code;
    }

    /**
     * 根据 异常信息 创建一个基础异常
     *
     * @param errorInfo 异常信息
     * @param args      消息参数
     */
    public BaseException(ErrorInfo errorInfo, Object... args) {
        this(errorInfo.getCode(), StrUtil.format(errorInfo.getMsg(), args));
        this.errorInfo = errorInfo;
    }

    @Override
    public String toString() {
        return StrUtil.format("{} (code={}, message={})", this.getClass().getSimpleName(), this.getCode(), super.getMessage());
    }

}
