package top.lshaci.framework.excel.handle;

import cn.hutool.core.io.IoUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import top.lshaci.framework.excel.entity.ImportSheetParam;
import top.lshaci.framework.excel.enums.ImportHandleErrorInfo;
import top.lshaci.framework.excel.exception.ImportHandleException;
import top.lshaci.framework.excel.service.impl.DefaultImportService;
import top.lshaci.framework.utils.FileTypeUtil;
import top.lshaci.framework.utils.enums.FileType;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

/**
 * Excel导入处理器
 *
 * @author lshaci
 * @since 1.0.2
 */
@Slf4j
public class ImportHandler {

	/**
	 * 允许的Excel文件类型
	 */
    private final static List<FileType> ALLOW_FILE_TYPES = Arrays.asList(
    		FileType.XLSX_DOCX, FileType.XLS_DOC,
    		FileType.WPS, FileType.WPSX
    );

    /**
     * 将输入流解析为指定对象类型的集合
     *
     * @param is 输入流
     * @param cls 解析对象类型
     * @return 解析后的对象集合
     */
    public static <E> List<E> parse(InputStream is, Class<E> cls) {
    	return parse(is, cls, null);
    }

    /**
     * 将输入流解析为指定对象类型的集合
     *
     * @param is 输入流
     * @param cls 解析对象类型
     * @param sheetParam Excel工作表参数
     * @return 解析后的对象集合
     */
	@SuppressWarnings("unchecked")
	public static <E> List<E> parse(InputStream is, Class<E> cls, ImportSheetParam sheetParam) {
		verifyParam(is, cls);
        ByteArrayOutputStream buffer = new ByteArrayOutputStream();
        IoUtil.copy(is, buffer);

        Workbook workbook = getWorkBook(buffer);
        return (List<E>) new DefaultImportService(cls, workbook, sheetParam).fetch();
	}

	/**
	 * 参数验证
	 *
     * @param is 输入流
     * @param cls 解析对象类型
	 */
	private static void verifyParam(InputStream is, Class<?> cls) {
		if (Objects.isNull(cls)) {
			throw new ImportHandleException(ImportHandleErrorInfo.entity_is_null);
		}
		if (Objects.isNull(is)) {
			throw new ImportHandleException(ImportHandleErrorInfo.data_is_null);
		}
	}

	/**
	 * 获取Excel工作簿
	 *
	 * @param buffer 缓冲流信息
	 * @return Excel工作簿
	 */
	private static Workbook getWorkBook(ByteArrayOutputStream buffer) {
		FileType fileType = getFileType(buffer);

        Workbook workbook = null;
        try (
        		InputStream is = new ByteArrayInputStream(buffer.toByteArray())
        ) {
            if (FileType.XLSX_DOCX.equals(fileType) || FileType.WPSX.equals(fileType)) {
                workbook = new XSSFWorkbook(is);
            }
            if (FileType.XLS_DOC.equals(fileType) || FileType.WPS.equals(fileType)) {
                workbook = new HSSFWorkbook(is);
            }
        } catch (Exception e) {
            log.error("创建Excel工作簿发生错误", e);
        }

        if (workbook == null) {
            throw new ImportHandleException(ImportHandleErrorInfo.workbook_is_null);
        }

        return workbook;
    }

	/**
	 * Excel文件类型
	 *
	 * @param buffer 缓冲流信息
	 * @return Excel文件类型
	 */
    private static FileType getFileType(ByteArrayOutputStream buffer) {
    	try (
    			InputStream is = new ByteArrayInputStream(buffer.toByteArray())
    	) {
    		FileType fileType = FileTypeUtil.getType(is);
    		if (ALLOW_FILE_TYPES.contains(fileType)) {
                return fileType;
            }
		} catch (Exception e) {
			log.error(ImportHandleErrorInfo.fetch_excel_type_error.getMsg(), e);
			throw new ImportHandleException(ImportHandleErrorInfo.fetch_excel_type_error);
		}

        throw new ImportHandleException(ImportHandleErrorInfo.not_excel);
    }
}
