package top.lshaci.framework.file.fdfs.config;

import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import top.lshaci.framework.file.fdfs.pool.TrackerServerPool;
import top.lshaci.framework.file.fdfs.properties.FrameworkFdfsProperties;
import top.lshaci.framework.file.fdfs.service.FdfsFileOperateService;
import top.lshaci.framework.file.fdfs.service.impl.FdfsFileOperateServiceImpl;
import top.lshaci.framework.file.properties.FrameworkFileProperties;

/**
 * FrameworkFdfsConfig
 *
 * @author lshaci
 * @since 1.0.9
 */
@Slf4j
@Configuration
@AllArgsConstructor
@ConditionalOnProperty(value = "framework.fdfs.enabled", matchIfMissing = true)
@EnableConfigurationProperties({FrameworkFileProperties.class, FrameworkFdfsProperties.class})
public class FrameworkFdfsConfig {

    private final FrameworkFdfsProperties frameworkFdfsProperties;

    private final FrameworkFileProperties frameworkFileProperties;

    /**
     * 创建FastDfs Tracker Server连接池
     *
     * @return FastDfs 服务连接池
     * @throws Exception 创建失败抛出异常
     */
    @Bean
    @ConditionalOnMissingBean
    public TrackerServerPool trackerServerPool() throws Exception {
        log.debug("Config fast dfs tracker server pool.");
        return new TrackerServerPool(frameworkFdfsProperties);
    }

    /**
     * 配置文件操作业务类
     *
     * @return 文件操作业务类
     * @throws Exception 创建失败抛出异常
     */
    @Bean
    public FdfsFileOperateService fdfsFileOperateService() throws Exception {
        log.debug("Config fastdfs file upload service, Max file size: {}...", frameworkFileProperties.getMaxSize());
        return new FdfsFileOperateServiceImpl(frameworkFileProperties.getMaxSize(), trackerServerPool());
    }

}
