package top.lshaci.framework.file.fdfs.pool;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.pool2.impl.GenericObjectPool;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.csource.fastdfs.ClientGlobal;
import org.csource.fastdfs.TrackerServer;
import top.lshaci.framework.file.fdfs.enums.FileFdfsErrorInfo;
import top.lshaci.framework.file.fdfs.exception.FileFdfsException;
import top.lshaci.framework.file.fdfs.factory.TrackerServerFactory;
import top.lshaci.framework.file.fdfs.properties.FrameworkFdfsProperties;

/**
 * <p>TrackerServerPool</p><br>
 *
 *
 * @author lshaci
 * @since 1.0.9
 */
@Slf4j
public class TrackerServerPool {

	/**
	 * The tracker server pool
	 */
	private final GenericObjectPool<TrackerServer> trackerServerPool;

    /**
     * Constructor a <code>TrackerServerPool</code> with fastDFS properties
     *
     * @param properties The fastDFS properties
     * @throws Exception if fastDFS client init failure
     */
	public TrackerServerPool(FrameworkFdfsProperties properties) throws Exception {
		// fast dfs client init
		ClientGlobal.initByProperties(properties.getFdfsProperties());
		log.debug("ClientGlobal configInfo: \n{}", ClientGlobal.configInfo());
		// pool config
		GenericObjectPoolConfig<TrackerServer> poolConfig = new GenericObjectPoolConfig<>();
		poolConfig.setMinIdle(properties.getMinIdle());
        poolConfig.setMaxIdle(properties.getMaxIdle());
		poolConfig.setMaxTotal(properties.getMinIdle());

		trackerServerPool = new GenericObjectPool<>(new TrackerServerFactory(), poolConfig);
	}

	/**
	 * Borrow tracker server from pool
	 *
	 * @return Tracker server instance
	 */
	public TrackerServer borrowObject() {
        try {
            return trackerServerPool.borrowObject();
        } catch (Exception e) {
        	log.error(FileFdfsErrorInfo.fetch_tracker_server_failed.getMsg(), e);
            throw new FileFdfsException(FileFdfsErrorInfo.fetch_tracker_server_failed);
        }
    }

    /**
     * Return tracker server to pool
     *
     * @param trackerServer the tracker server instance
     */
    public void returnObject(TrackerServer trackerServer){
    	if (trackerServer != null) {
    		trackerServerPool.returnObject(trackerServer);
		}
    }

}
