package top.lshaci.framework.file.minio.service;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.StrUtil;
import io.minio.GetObjectArgs;
import io.minio.RemoveObjectsArgs;
import io.minio.http.Method;
import io.minio.messages.DeleteObject;
import top.lshaci.framework.file.service.FileOperateService;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.time.Duration;
import java.util.List;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * MinioFileOperateService
 *
 * @author lshaci
 * @since 1.1.0
 */
public interface MinioFileOperateService extends FileOperateService {

    /**
     * min io 中 bucket 的名称
     *
     * @return bucket <b>framework.minio.bucket</b>
     */
    String bucket();

    /**
     * 上传对象到 min io 中
     *
     * @param inputStream 对象的输入流
     * @param extName     对象扩展名(不含<b>.</b>)
     * @return min io 中的文件名称
     * @throws IOException 如果产生 I/O 异常
     */
    @Override
    default String upload(InputStream inputStream, String extName) throws IOException {
        return single(inputStream, bucket(), () -> StrUtil.format("{}{}{}", IdUtil.objectId(), StrUtil.DOT, extName));
    }

    /**
     * 上传对象到 min io 中
     *
     * @param inputStream  对象的输入流
     * @param pathSupplier 对象地址生成方式
     * @return min io 中的文件名称
     * @throws IOException 如果产生 I/O 异常
     */
    default String single(InputStream inputStream, Supplier<String> pathSupplier) throws IOException {
        return single(inputStream, bucket(), pathSupplier);
    };

    /**
     * 上传对象到 min io 中
     *
     * @param inputStream  对象的输入流
     * @param bucket       min io 中的 bucket
     * @param pathSupplier 对象地址生成方式
     * @return min io 中的文件名称
     * @throws IOException 如果产生 I/O 异常
     */
    String single(InputStream inputStream, String bucket, Supplier<String> pathSupplier) throws IOException;

    /**
     * 根据对象地址, 删除min io 中的对象
     *
     * @param name min io 中的对象名称
     */
    @Override
    default void delete(String name) {
        delete(bucket(), name);
    }

    /**
     * 根据对象地址集合删除 oss 中多个对象
     *
     * @param names min io 中的对象名称集合
     */
    @Override
    default void delete(List<String> names) {
        delete(bucket(), names);
    }

    /**
     * 根据对象地址, 删除min io 中指定 bucket 中的对象
     *
     * @param bucket min io 中的 bucket
     * @param name   min io 中的对象名称
     */
    default void delete(String bucket, String name) {
        delete(bucket, CollUtil.newArrayList(name));
    }

    /**
     * 根据对象地址集合, 删除min io 中指定 bucket 中多个对象
     *
     * @param bucket min io 中的 bucket
     * @param names  min io 中的对象名称集合
     */
    default void delete(String bucket, List<String> names) {
        if (CollUtil.isEmpty(names)) {
            return;
        }
        List<DeleteObject> deleteObjects = names.stream().map(DeleteObject::new).collect(Collectors.toList());
        RemoveObjectsArgs args = RemoveObjectsArgs.builder().bucket(bucket).objects(deleteObjects).build();
        delete(args);
    }

    /**
     * 根据删除对象请求, 删除 min io 中的对象
     *
     * @param args 删除对象请求
     */
    void delete(RemoveObjectsArgs args);

    /**
     * 根据对象地址, 在min io 中或获取对象 指定 bucket 中的对象
     *
     * @param name         min io 中的对象地址
     * @param outputStream 输出流
     */
    @Override
    default void write(String name, OutputStream outputStream) {
        write(bucket(), name, outputStream);
    }

    /**
     * 根据对象地址, 在min io 中指定 bucket 中或获取对象
     *
     * @param bucket       min io 中的 bucket
     * @param name         min io 中的对象名称
     * @param outputStream 输出流
     */
    default void write(String bucket, String name, OutputStream outputStream) {
        final GetObjectArgs args = GetObjectArgs.builder().bucket(bucket).object(name).build();
        write(args, outputStream);
    }

    /**
     * 根据获取对象请求, 在 min io 中获取对象
     *
     * @param args         获取对象请求
     * @param outputStream 输出流
     */
    void write(GetObjectArgs args, OutputStream outputStream);

    /**
     * 获取 GET 的预签名地址
     *
     * @param bucket min io 中的 bucket
     * @param name   min io 中的对象名称
     * @param method 地址请求方式
     * @param expiry 失效时间
     * @return 预签名地址
     */
    String presignedUrl(String bucket, String name, Method method, Duration expiry);

    /**
     * 获取 GET 的预签名地址
     *
     * @param name min io 中的对象名称
     * @return GET 的预签名地址
     */
    default String getPresignedUrl(String name) {
        return presignedUrl(bucket(), name, Method.GET, null);
    }

}
