package top.lshaci.framework.file.minio.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.io.IoUtil;
import cn.hutool.core.util.StrUtil;
import io.minio.*;
import io.minio.errors.MinioException;
import io.minio.http.Method;
import io.minio.messages.DeleteObject;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.unit.DataSize;
import top.lshaci.framework.file.enums.FileErrorInfo;
import top.lshaci.framework.file.exception.FileDownloadException;
import top.lshaci.framework.file.exception.FileOperateException;
import top.lshaci.framework.file.exception.FileUploadException;
import top.lshaci.framework.file.minio.enums.FileMinioErrorInfo;
import top.lshaci.framework.file.minio.exception.FileMinioException;
import top.lshaci.framework.file.minio.service.MinioFileOperateService;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.GeneralSecurityException;
import java.time.Duration;
import java.util.Optional;
import java.util.function.Supplier;

/**
 * OssFileOperateServiceImpl
 *
 * @author lshaci
 * @since 1.1.0
 */
@Slf4j
@AllArgsConstructor
public class MinioFileOperateServiceImpl implements MinioFileOperateService {

    private final MinioClient minioClient;

    private final Duration defaultExpiry;

    private final String defaultBucket;

    private final DataSize maxSize;

    @Override
    public String single(InputStream inputStream, String bucket, Supplier<String> pathSupplier) throws IOException {
        final int available = inputStream.available();
        if (available <= 0) {
            throw new FileUploadException(FileErrorInfo.size_is_empty);
        }
        log.debug("Upload size: {}, Max size: {}", available, maxSize);
        if (available > maxSize.toBytes()) {
            throw new FileUploadException(FileErrorInfo.exceed_limit);
        }
        final String name = pathSupplier.get();
        verifyBucketAndName(bucket, name);
        try {
            minioClient.putObject(PutObjectArgs.builder()
                    .bucket(bucket)
                    .object(name)
                    .stream(inputStream, inputStream.available(), -1)
                    .build());
        } catch (MinioException | GeneralSecurityException e) {
            log.error(FileErrorInfo.upload_failed.getMsg(), e);
            throw new FileUploadException(FileErrorInfo.upload_failed);
        }

        log.debug("Upload success, Object name is: {}", name);
        return name;
    }

    @Override
    public void delete(RemoveObjectsArgs args) {
        Iterable<DeleteObject> deleteObjects = args.objects();
        if (CollUtil.isEmpty(deleteObjects)) {
            log.warn("未指定需要删除的文件");
            return;
        }
        verifyBucket(args.bucket());
        minioClient.removeObjects(args);
    }

    @Override
    public void write(GetObjectArgs args, OutputStream outputStream) {
        verifyBucketAndName(args.bucket(), args.object());
        try {
            GetObjectResponse object = minioClient.getObject(args);
            IoUtil.copy(object, outputStream);
        } catch (Exception e) {
            log.error(FileErrorInfo.fetch_failed.getMsg(), e);
            throw new FileDownloadException(FileErrorInfo.fetch_failed);
        }
    }

    @Override
    public String presignedUrl(String bucket, String name, Method method, Duration expiry) {
        verifyBucketAndName(bucket, name);
        try {
            return minioClient.getPresignedObjectUrl(GetPresignedObjectUrlArgs.builder()
                    .bucket(bucket)
                    .object(name)
                    .method(Optional.ofNullable(method).orElse(Method.GET))
                    .expiry((int) Optional.ofNullable(expiry).orElse(defaultExpiry).getSeconds())
                    .build()
            );
        } catch (IOException | MinioException | GeneralSecurityException e) {
            log.error(FileMinioErrorInfo.presigned_url_fetch_failure.getMsg(), e);
            throw new FileMinioException(FileMinioErrorInfo.presigned_url_fetch_failure);
        }
    }

    @Override
    public String bucket() {
        return this.defaultBucket;
    }

    /**
     * 验证 bucket 是否为空
     *
     * @param bucket 存储的 bucket
     */
    private void verifyBucket(String bucket) {
        if (StrUtil.isBlank(bucket)) {
            throw new FileMinioException(FileMinioErrorInfo.bucket_is_blank);
        }
        try {
            boolean bucketExists = minioClient.bucketExists(BucketExistsArgs.builder().bucket(bucket).build());
            if (!bucketExists) {
                minioClient.makeBucket(MakeBucketArgs.builder().bucket(bucket).build());
            }
        } catch (IOException | MinioException | GeneralSecurityException e) {
            log.error(FileMinioErrorInfo.bucket_create_failure.getMsg(), e);
            throw new FileMinioException(FileMinioErrorInfo.bucket_create_failure);
        }
    }

    /**
     * 验证 key 是否为空
     *
     * @param key 文件的 key
     */
    private void verifyName(String key) {
        if (StrUtil.isBlank(key)) {
            throw new FileOperateException(FileErrorInfo.path_is_blank);
        }
    }

    /**
     * 验证 bucket 和 name 是否为空
     *
     * @param bucket 存储的 bucket
     * @param name   文件的 name
     */
    private void verifyBucketAndName(String bucket, String name) {
        verifyName(name);
        verifyBucket(bucket);
    }
}
