package top.lshaci.framework.log.aspect;

import cn.hutool.log.LogFactory;
import lombok.AllArgsConstructor;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.core.annotation.Order;
import top.lshaci.framework.log.annotation.Log;
import top.lshaci.framework.log.enums.OperateStatus;
import top.lshaci.framework.log.model.LogModel;
import top.lshaci.framework.log.service.LogService;
import top.lshaci.framework.log.thread.LogThreadLocal;

import java.lang.reflect.Method;

/**
 * LogAspect
 *
 * <pre>
 *     1.1.0: 添加 module 属性，修复无返回值bug
 * </pre>
 *
 * @author lshaci
 * @since 1.1.0
 */
@Aspect
@Order(2)
@AllArgsConstructor
public class LogAspect {

    private static final cn.hutool.log.Log logger = LogFactory.get(LogAspect.class);

    private final LogService logService;

    /**
     * 切入点进行日志操作
     *
     * @param joinPoint 切入点信息
     * @param log       日志标题
     * @return 目标方法的返回值
     * @throws Throwable 目标方法抛出的异常
     */
    @Around("@annotation(log)")
    public Object around(ProceedingJoinPoint joinPoint, Log log) throws Throwable {
        LogModel logModel = new LogModel();
        try {
            fill(joinPoint, log, logModel);
            Object result = joinPoint.proceed();
            logModel.setStatus(OperateStatus.success);
            return result;
        } catch (Throwable e) {
            logModel.setStatus(OperateStatus.failure)
                    .setExceptionMessage(e.getMessage());
            throw e;
        } finally {
            handle(logModel);
        }
    }

    /**
     * 填充日志对象信息
     *
     * @param joinPoint 切入点信息
     * @param log       日志标题
     * @param logModel  日志信息
     */
    private void fill(ProceedingJoinPoint joinPoint, Log log, LogModel logModel) {
        MethodSignature methodSignature = (MethodSignature) joinPoint.getSignature();
        Method targetMethod = methodSignature.getMethod();
        logModel.setModule(log.module()).setTitle(log.title())
                .setTargetClass(targetMethod.getDeclaringClass())
                .setTargetMethod(targetMethod.getName()).setArgs(joinPoint.getArgs());
    }

    /**
     * 日志处理
     *
     * @param logModel 日志信息
     */
    private void handle(LogModel logModel) {
        try {
            Object content = LogThreadLocal.get(Object.class);
            logModel.setContent(content);
            logService.record(logModel);
        } catch (Exception e) {
            logger.error(e, "记录日志失败");
        } finally {
            LogThreadLocal.remove();
        }
    }
}
